<?php
// admin_aniversarios.php - SISTEMA DE GERENCIAMENTO DE ANIVERSÁRIOS
require_once 'config.php';
verificarRole(['admin']);

$admin_id = $_SESSION['user_id'];
$mensagem_sucesso = '';
$mensagem_erro = '';

// Criar diretório para imagens de aniversário se não existir
$aniversarios_dir = 'uploads/aniversarios';
if (!file_exists($aniversarios_dir)) {
    mkdir($aniversarios_dir, 0755, true);
}

// Buscar configurações de aniversários
try {
    $stmt = $pdo->query("SELECT * FROM configuracoes_aniversarios LIMIT 1");
    $config_aniv = $stmt->fetch();

    if (!$config_aniv) {
        // Criar configuração padrão se não existir
        $pdo->exec("INSERT INTO configuracoes_aniversarios (mensagem_padrao_assunto, mensagem_padrao_corpo) VALUES ('Parabéns {nome} pelo seu dia! 🎉', 'Olá {nome_completo}...')");
        $stmt = $pdo->query("SELECT * FROM configuracoes_aniversarios LIMIT 1");
        $config_aniv = $stmt->fetch();
    }
} catch (PDOException $e) {
    $mensagem_erro = "Erro ao carregar configurações. Execute a migration_aniversarios.sql";
    $config_aniv = null;
}

// Processar envio de mensagem personalizada
if (isset($_POST['enviar_mensagem'])) {
    $convidado_id = intval($_POST['convidado_id']);
    $tipo_envio = $_POST['tipo_envio']; // 'email', 'whatsapp', 'ambos'
    $mensagem_personalizada = $_POST['mensagem_personalizada'];
    $ano_atual = date('Y');

    try {
        // Buscar dados do convidado
        $stmt = $pdo->prepare("SELECT * FROM convidados WHERE id = ?");
        $stmt->execute([$convidado_id]);
        $convidado = $stmt->fetch();

        if ($convidado) {
            // Substituir campos dinâmicos
            $nome_partes = explode(' ', trim($convidado['nome']));
            $primeiro_nome = $nome_partes[0];
            $sobrenome = count($nome_partes) > 1 ? end($nome_partes) : '';

            $mensagem_final = str_replace(
                ['{nome}', '{sobrenome}', '{nome_completo}'],
                [$primeiro_nome, $sobrenome, $convidado['nome']],
                $mensagem_personalizada
            );

            $enviou = false;

            // Enviar por email
            if ($tipo_envio === 'email' || $tipo_envio === 'ambos') {
                if (!empty($convidado['email']) && filter_var($convidado['email'], FILTER_VALIDATE_EMAIL)) {
                    // Preparar assunto
                    $assunto = $config_aniv['mensagem_padrao_assunto'];
                    $assunto = str_replace(
                        ['{nome}', '{sobrenome}', '{nome_completo}'],
                        [$primeiro_nome, $sobrenome, $convidado['nome']],
                        $assunto
                    );

                    // Preparar headers do email
                    $headers = "MIME-Version: 1.0\r\n";
                    $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
                    $headers .= "From: " . EMAIL_NOME_REMETENTE . " <" . EMAIL_REMETENTE . ">\r\n";

                    // Converter mensagem para HTML
                    $corpo_html = nl2br(htmlspecialchars($mensagem_final));

                    // HTML do email
                    $html_email = "
                    <!DOCTYPE html>
                    <html>
                    <head>
                        <meta charset='UTF-8'>
                        <style>
                            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                            .header { background: linear-gradient(135deg, #FFD700 0%, #FFA500 100%); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
                            .content { background: #fff; padding: 30px; border: 1px solid #ddd; }
                            .footer { background: #f5f5f5; padding: 20px; text-align: center; font-size: 12px; color: #666; border-radius: 0 0 10px 10px; }
                            .emoji { font-size: 48px; }
                        </style>
                    </head>
                    <body>
                        <div class='container'>
                            <div class='header'>
                                <div class='emoji'>🎉</div>
                                <h1 style='margin: 10px 0 0 0;'>{$assunto}</h1>
                            </div>
                            <div class='content'>
                                {$corpo_html}
                            </div>
                            <div class='footer'>
                                Enviado por " . EMAIL_NOME_REMETENTE . "
                            </div>
                        </div>
                    </body>
                    </html>";

                    // Enviar email
                    if (mail($convidado['email'], $assunto, $html_email, $headers)) {
                        $enviou = true;
                        error_log("✅ Email de aniversário enviado para: " . $convidado['email']);
                    } else {
                        error_log("❌ Erro ao enviar email de aniversário para: " . $convidado['email']);
                    }
                }
            }

            // Enviar por WhatsApp (abre link)
            if ($tipo_envio === 'whatsapp' || $tipo_envio === 'ambos') {
                if (!empty($convidado['whatsapp'])) {
                    $enviou = true;
                }
            }

            if ($enviou) {
                // Registrar no histórico
                $stmt = $pdo->prepare("
                    INSERT INTO aniversarios_enviados (convidado_id, ano, tipo_envio, enviado_por, mensagem_enviada, enviado_automatico)
                    VALUES (?, ?, ?, ?, ?, 0)
                    ON DUPLICATE KEY UPDATE
                        tipo_envio = VALUES(tipo_envio),
                        mensagem_enviada = VALUES(mensagem_enviada),
                        created_at = CURRENT_TIMESTAMP
                ");
                $stmt->execute([$convidado_id, $ano_atual, $tipo_envio, $admin_id, $mensagem_final]);

                // Criar notificação
                criarNotificacao([
                    'user_id' => $admin_id,
                    'tipo' => 'sistema',
                    'titulo' => 'Mensagem de aniversário enviada',
                    'mensagem' => "Mensagem enviada para {$convidado['nome']} via {$tipo_envio}",
                    'referencia_tipo' => 'aniversario',
                    'referencia_id' => $convidado_id
                ]);

                $mensagem_sucesso = "Mensagem enviada com sucesso para {$convidado['nome']}!";
            }
        }
    } catch (PDOException $e) {
        $mensagem_erro = "Erro ao enviar mensagem: " . $e->getMessage();
    }
}

// Processar atualização de configurações
if (isset($_POST['salvar_config'])) {
    $assunto_padrao = htmlspecialchars($_POST['mensagem_padrao_assunto'], ENT_QUOTES, 'UTF-8');
    $corpo_padrao = htmlspecialchars($_POST['mensagem_padrao_corpo'], ENT_QUOTES, 'UTF-8');
    $whatsapp_padrao = htmlspecialchars($_POST['mensagem_whatsapp_padrao'], ENT_QUOTES, 'UTF-8');
    $envio_automatico = isset($_POST['envio_automatico_ativo']) ? 1 : 0;

    $imagem_path = $config_aniv['imagem_anexo'];

    // Upload de imagem
    if (isset($_FILES['imagem_anexo']) && $_FILES['imagem_anexo']['error'] == 0) {
        $extensao = pathinfo($_FILES['imagem_anexo']['name'], PATHINFO_EXTENSION);
        $nome_arquivo = 'aniversario_' . time() . '.' . $extensao;
        $caminho_completo = $aniversarios_dir . '/' . $nome_arquivo;

        if (move_uploaded_file($_FILES['imagem_anexo']['tmp_name'], $caminho_completo)) {
            // Remover imagem antiga
            if ($imagem_path && file_exists($imagem_path)) {
                unlink($imagem_path);
            }
            $imagem_path = $caminho_completo;
        }
    }

    try {
        $stmt = $pdo->prepare("
            UPDATE configuracoes_aniversarios
            SET mensagem_padrao_assunto = ?,
                mensagem_padrao_corpo = ?,
                mensagem_whatsapp_padrao = ?,
                imagem_anexo = ?,
                envio_automatico_ativo = ?
            WHERE id = ?
        ");
        $stmt->execute([
            $assunto_padrao,
            $corpo_padrao,
            $whatsapp_padrao,
            $imagem_path,
            $envio_automatico,
            $config_aniv['id']
        ]);

        $mensagem_sucesso = "Configurações salvas com sucesso!";

        // Recarregar configurações
        $stmt = $pdo->query("SELECT * FROM configuracoes_aniversarios LIMIT 1");
        $config_aniv = $stmt->fetch();
    } catch (PDOException $e) {
        $mensagem_erro = "Erro ao salvar: " . $e->getMessage();
    }
}

// Filtros
$filtro_mes = isset($_GET['mes']) ? intval($_GET['mes']) : intval(date('m'));
$filtro_dia = isset($_GET['dia']) ? intval($_GET['dia']) : 0;
$visualizar = isset($_GET['visualizar']) ? $_GET['visualizar'] : 'pendentes'; // 'pendentes' ou 'historico'

$ano_atual = date('Y');

// Buscar aniversariantes do mês (que ainda não receberam mensagem este ano)
// NOTA: data_nascimento armazena apenas DIA e MÊS (sem ano)
if ($visualizar === 'pendentes') {
    $query = "
        SELECT c.*,
               DAY(c.data_nascimento) as dia_aniv,
               MONTH(c.data_nascimento) as mes_aniv,
               ae.id as ja_enviado
        FROM convidados c
        LEFT JOIN aniversarios_enviados ae ON c.id = ae.convidado_id AND ae.ano = ?
        WHERE c.data_nascimento IS NOT NULL
          AND MONTH(c.data_nascimento) = ?
          AND ae.id IS NULL
    ";

    $params = [$ano_atual, $filtro_mes];

    if ($filtro_dia > 0) {
        $query .= " AND DAY(c.data_nascimento) = ?";
        $params[] = $filtro_dia;
    }

    $query .= " ORDER BY DAY(c.data_nascimento) ASC, c.nome ASC";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $aniversariantes = $stmt->fetchAll();

} else {
    // Buscar histórico (já enviados este ano)
    $query = "
        SELECT c.*,
               DAY(c.data_nascimento) as dia_aniv,
               MONTH(c.data_nascimento) as mes_aniv,
               ae.created_at as data_envio,
               ae.tipo_envio,
               ae.enviado_automatico,
               u.nome as enviado_por_nome
        FROM convidados c
        INNER JOIN aniversarios_enviados ae ON c.id = ae.convidado_id AND ae.ano = ?
        LEFT JOIN users u ON ae.enviado_por = u.id
        WHERE c.data_nascimento IS NOT NULL
          AND MONTH(c.data_nascimento) = ?
    ";

    $params = [$ano_atual, $filtro_mes];

    if ($filtro_dia > 0) {
        $query .= " AND DAY(c.data_nascimento) = ?";
        $params[] = $filtro_dia;
    }

    $query .= " ORDER BY ae.created_at DESC";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $aniversariantes = $stmt->fetchAll();
}

// Contador de aniversários do mês (pendentes)
$stmt = $pdo->prepare("
    SELECT COUNT(*) as total
    FROM convidados c
    LEFT JOIN aniversarios_enviados ae ON c.id = ae.convidado_id AND ae.ano = ?
    WHERE c.data_nascimento IS NOT NULL
      AND MONTH(c.data_nascimento) = ?
      AND ae.id IS NULL
");
$stmt->execute([$ano_atual, $filtro_mes]);
$total_pendentes_mes = $stmt->fetch()['total'];

// Contador de aniversários de HOJE (pendentes)
$stmt = $pdo->prepare("
    SELECT COUNT(*) as total
    FROM convidados c
    LEFT JOIN aniversarios_enviados ae ON c.id = ae.convidado_id AND ae.ano = ?
    WHERE c.data_nascimento IS NOT NULL
      AND MONTH(c.data_nascimento) = ?
      AND DAY(c.data_nascimento) = ?
      AND ae.id IS NULL
");
$stmt->execute([$ano_atual, intval(date('m')), intval(date('d'))]);
$total_hoje = $stmt->fetch()['total'];

$meses = [
    1 => 'Janeiro', 2 => 'Fevereiro', 3 => 'Março', 4 => 'Abril',
    5 => 'Maio', 6 => 'Junho', 7 => 'Julho', 8 => 'Agosto',
    9 => 'Setembro', 10 => 'Outubro', 11 => 'Novembro', 12 => 'Dezembro'
];
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Aniversários - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/dashboard-style.css">
    <link rel="stylesheet" href="assets/css/sidebar.css">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }

        :root[data-theme="dark"] {
            --bg-primary: #000000;
            --bg-secondary: #1a1a1a;
            --bg-card: #2a2a2a;
            --color-primary: #FFD700;
            --color-text: #ffffff;
            --color-text-secondary: #999999;
            --border-color: #333333;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--bg-secondary);
            color: var(--color-text);
        }

        .container {
            max-width: 1600px;
            margin: 30px auto;
            padding: 0 20px;
        }

        .page-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            flex-wrap: wrap;
            gap: 20px;
        }

        .page-title {
            font-size: 32px;
            color: var(--color-primary);
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .budget-badge {
            display: inline-flex;
            align-items: center;
            gap: 10px;
            background: linear-gradient(135deg, #FF6B6B 0%, #FF8E53 100%);
            color: white;
            padding: 10px 20px;
            border-radius: 25px;
            font-size: 16px;
            font-weight: bold;
            box-shadow: 0 4px 15px rgba(255, 107, 107, 0.3);
        }

        .budget-number {
            background: rgba(255, 255, 255, 0.3);
            padding: 5px 15px;
            border-radius: 15px;
            font-size: 20px;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: var(--bg-card);
            padding: 25px;
            border-radius: 12px;
            border: 1px solid var(--border-color);
            transition: all 0.3s;
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.5);
        }

        .stat-icon {
            font-size: 36px;
            margin-bottom: 10px;
        }

        .stat-value {
            font-size: 32px;
            font-weight: bold;
            color: var(--color-primary);
            margin-bottom: 5px;
        }

        .stat-label {
            color: var(--color-text-secondary);
            font-size: 14px;
        }

        .section-card {
            background: var(--bg-card);
            padding: 25px;
            border-radius: 12px;
            margin-bottom: 30px;
            border: 1px solid var(--border-color);
        }

        .section-title {
            font-size: 24px;
            color: var(--color-primary);
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .filters-section {
            background: var(--bg-card);
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 30px;
            border: 1px solid var(--border-color);
        }

        .filters-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }

        .filter-group {
            display: flex;
            flex-direction: column;
        }

        .filter-group label {
            color: var(--color-text-secondary);
            margin-bottom: 5px;
            font-size: 14px;
        }

        .filter-group input,
        .filter-group select,
        .filter-group textarea {
            padding: 10px;
            border: 1px solid var(--border-color);
            border-radius: 8px;
            background: var(--bg-secondary);
            color: var(--color-text);
            font-family: inherit;
        }

        .filter-group textarea {
            min-height: 120px;
            resize: vertical;
        }

        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            text-decoration: none;
            font-size: 14px;
            transition: all 0.3s;
            display: inline-flex;
            align-items: center;
            gap: 8px;
            font-family: inherit;
        }

        .btn-primary {
            background: var(--color-primary);
            color: #000;
        }

        .btn-success {
            background: #4CAF50;
            color: white;
        }

        .btn-info {
            background: #17a2b8;
            color: white;
        }

        .btn-warning {
            background: #FF9800;
            color: white;
        }

        .btn-secondary {
            background: var(--bg-card);
            color: var(--color-text);
            border: 1px solid var(--border-color);
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(255, 215, 0, 0.2);
        }

        .table {
            width: 100%;
            border-collapse: collapse;
        }

        .table th {
            background: var(--bg-secondary);
            padding: 12px;
            text-align: left;
            color: var(--color-primary);
            border-bottom: 2px solid var(--border-color);
        }

        .table td {
            padding: 12px;
            border-bottom: 1px solid var(--border-color);
        }

        .table tr:hover {
            background: rgba(255, 215, 0, 0.05);
        }

        .whatsapp-btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            width: 40px;
            height: 40px;
            background: #25D366;
            color: white;
            border-radius: 50%;
            font-size: 22px;
            text-decoration: none;
            transition: all 0.3s;
        }

        .whatsapp-btn:hover {
            transform: scale(1.1);
            box-shadow: 0 4px 15px rgba(37, 211, 102, 0.4);
        }

        .email-btn {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            width: 40px;
            height: 40px;
            background: #EA4335;
            color: white;
            border-radius: 50%;
            font-size: 18px;
            text-decoration: none;
            transition: all 0.3s;
            border: none;
            cursor: pointer;
        }

        .email-btn:hover {
            transform: scale(1.1);
            box-shadow: 0 4px 15px rgba(234, 67, 53, 0.4);
        }

        .badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: bold;
        }

        .badge-today {
            background: rgba(255, 107, 107, 0.2);
            color: #FF6B6B;
        }

        .badge-upcoming {
            background: rgba(76, 175, 80, 0.2);
            color: #4CAF50;
        }

        .badge-auto {
            background: rgba(23, 162, 184, 0.2);
            color: #17a2b8;
        }

        .badge-manual {
            background: rgba(255, 152, 0, 0.2);
            color: #FF9800;
        }

        .alert {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }

        .alert-success {
            background: rgba(76, 175, 80, 0.2);
            color: #4CAF50;
            border: 1px solid #4CAF50;
        }

        .alert-error {
            background: rgba(244, 67, 54, 0.2);
            color: #F44336;
            border: 1px solid #F44336;
        }

        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
        }

        .modal-content {
            background: var(--bg-card);
            margin: 5% auto;
            padding: 30px;
            border: 1px solid var(--border-color);
            border-radius: 12px;
            width: 90%;
            max-width: 600px;
        }

        .close {
            color: var(--color-text-secondary);
            float: right;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
        }

        .close:hover {
            color: var(--color-primary);
        }

        .tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            border-bottom: 2px solid var(--border-color);
        }

        .tab {
            padding: 12px 24px;
            background: transparent;
            border: none;
            color: var(--color-text-secondary);
            cursor: pointer;
            border-bottom: 3px solid transparent;
            transition: all 0.3s;
        }

        .tab.active {
            color: var(--color-primary);
            border-bottom-color: var(--color-primary);
        }

        .tab:hover {
            color: var(--color-primary);
        }

        .campos-dinamicos {
            background: rgba(255, 215, 0, 0.1);
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 15px;
            border: 1px solid var(--color-primary);
        }

        .campos-dinamicos strong {
            color: var(--color-primary);
        }

        .campos-dinamicos code {
            background: var(--bg-secondary);
            padding: 2px 8px;
            border-radius: 4px;
            font-family: 'Courier New', monospace;
        }

        @media (max-width: 768px) {
            .container {
                padding: 10px;
            }

            .page-header {
                flex-direction: column;
                align-items: flex-start;
            }

            .table {
                font-size: 12px;
            }

            .table th, .table td {
                padding: 8px 4px;
            }
        }
    </style>
</head>
<body>
    <?php include 'assets/sidebar.php'; ?>

    <div class="container">
        <div class="page-header">
            <h1 class="page-title">
                <i class="fas fa-birthday-cake"></i>
                Gerenciamento de Aniversários
            </h1>
            <div class="budget-badge">
                <i class="fas fa-gift"></i>
                <span>Aniversários de <?= $meses[$filtro_mes] ?>:</span>
                <span class="budget-number"><?= $total_pendentes_mes ?></span>
            </div>
        </div>

        <?php if ($mensagem_sucesso): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i> <?= htmlspecialchars($mensagem_sucesso) ?>
            </div>
        <?php endif; ?>

        <?php if ($mensagem_erro): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-triangle"></i> <?= htmlspecialchars($mensagem_erro) ?>
            </div>
        <?php endif; ?>

        <!-- Cards de Estatísticas -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon" style="color: #FF6B6B;"><i class="fas fa-calendar-day"></i></div>
                <div class="stat-value"><?= $total_hoje ?></div>
                <div class="stat-label">Aniversariantes Hoje</div>
            </div>

            <div class="stat-card">
                <div class="stat-icon" style="color: #4CAF50;"><i class="fas fa-calendar-check"></i></div>
                <div class="stat-value"><?= $total_pendentes_mes ?></div>
                <div class="stat-label">Pendentes em <?= $meses[$filtro_mes] ?></div>
            </div>

            <div class="stat-card">
                <div class="stat-icon" style="color: #17a2b8;"><i class="fas fa-robot"></i></div>
                <div class="stat-value"><?= $config_aniv && $config_aniv['envio_automatico_ativo'] ? 'Ativo' : 'Inativo' ?></div>
                <div class="stat-label">Envio Automático</div>
            </div>
        </div>

        <!-- Seção de Filtros -->
        <div class="filters-section">
            <form method="GET" action="">
                <input type="hidden" name="visualizar" value="<?= htmlspecialchars($visualizar) ?>">
                <div class="filters-grid">
                    <div class="filter-group">
                        <label>Mês</label>
                        <select name="mes">
                            <?php foreach ($meses as $num => $nome): ?>
                                <option value="<?= $num ?>" <?= $filtro_mes == $num ? 'selected' : '' ?>><?= $nome ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="filter-group">
                        <label>Dia (Opcional)</label>
                        <select name="dia">
                            <option value="0">Todos os dias</option>
                            <?php for ($d = 1; $d <= 31; $d++): ?>
                                <option value="<?= $d ?>" <?= $filtro_dia == $d ? 'selected' : '' ?>><?= $d ?></option>
                            <?php endfor; ?>
                        </select>
                    </div>
                </div>
                <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                    <button type="submit" class="btn btn-primary"><i class="fas fa-filter"></i> Filtrar</button>
                    <a href="?mes=<?= date('m') ?>" class="btn btn-secondary"><i class="fas fa-redo"></i> Limpar</a>
                    <button type="button" class="btn btn-info" onclick="document.getElementById('configModal').style.display='block'">
                        <i class="fas fa-cog"></i> Configurar Mensagem Padrão
                    </button>
                </div>
            </form>
        </div>

        <!-- Tabs de Navegação -->
        <div class="tabs">
            <a href="?mes=<?= $filtro_mes ?>&visualizar=pendentes" class="tab <?= $visualizar === 'pendentes' ? 'active' : '' ?>">
                <i class="fas fa-clock"></i> Pendentes
            </a>
            <a href="?mes=<?= $filtro_mes ?>&visualizar=historico" class="tab <?= $visualizar === 'historico' ? 'active' : '' ?>">
                <i class="fas fa-history"></i> Histórico
            </a>
        </div>

        <!-- Lista de Aniversariantes -->
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-users"></i>
                <?= $visualizar === 'pendentes' ? 'Aniversariantes Pendentes' : 'Histórico de Envios' ?>
            </h2>

            <?php if (count($aniversariantes) > 0): ?>
                <table class="table">
                    <thead>
                        <tr>
                            <th>Nome</th>
                            <th>Aniversário (Dia/Mês)</th>
                            <?php if ($visualizar === 'pendentes'): ?>
                                <th>Email</th>
                                <th>WhatsApp</th>
                                <th>Ações</th>
                            <?php else: ?>
                                <th>Enviado em</th>
                                <th>Tipo</th>
                                <th>Enviado por</th>
                            <?php endif; ?>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($aniversariantes as $aniv): ?>
                        <tr>
                            <td>
                                <?= htmlspecialchars($aniv['nome']) ?>
                                <?php if ($visualizar === 'pendentes'): ?>
                                    <?php
                                    $hoje = date('d');
                                    $mes_hoje = date('m');
                                    if ($aniv['dia_aniv'] == $hoje && $aniv['mes_aniv'] == $mes_hoje):
                                    ?>
                                        <span class="badge badge-today">HOJE! 🎉</span>
                                    <?php elseif ($aniv['dia_aniv'] > $hoje && $aniv['mes_aniv'] == $mes_hoje): ?>
                                        <span class="badge badge-upcoming">Em breve</span>
                                    <?php endif; ?>
                                <?php endif; ?>
                            </td>
                            <td>
                                <?php
                                // Mostrar apenas dia/mês sem ano
                                if ($aniv['data_nascimento']) {
                                    echo str_pad($aniv['dia_aniv'], 2, '0', STR_PAD_LEFT) . '/' . str_pad($aniv['mes_aniv'], 2, '0', STR_PAD_LEFT);
                                } else {
                                    echo '-';
                                }
                                ?>
                            </td>

                            <?php if ($visualizar === 'pendentes'): ?>
                                <td><?= htmlspecialchars($aniv['email'] ?: '-') ?></td>
                                <td>
                                    <?php if ($aniv['whatsapp']): ?>
                                        <?php
                                        $telefone_limpo = preg_replace('/[^0-9]/', '', $aniv['whatsapp']);
                                        if (!str_starts_with($telefone_limpo, '55')) {
                                            $telefone_limpo = '55' . $telefone_limpo;
                                        }
                                        ?>
                                        <a href="https://wa.me/<?= $telefone_limpo ?>" target="_blank" class="whatsapp-btn" title="WhatsApp">
                                            <i class="fab fa-whatsapp"></i>
                                        </a>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <button onclick="abrirModalEnvio(<?= $aniv['id'] ?>, '<?= htmlspecialchars(addslashes($aniv['nome'])) ?>')" class="email-btn" title="Enviar Mensagem">
                                        <i class="fas fa-paper-plane"></i>
                                    </button>
                                </td>
                            <?php else: ?>
                                <td><?= date('d/m/Y H:i', strtotime($aniv['data_envio'])) ?></td>
                                <td>
                                    <span class="badge <?= $aniv['enviado_automatico'] ? 'badge-auto' : 'badge-manual' ?>">
                                        <?= $aniv['tipo_envio'] ?>
                                        <?= $aniv['enviado_automatico'] ? '(Auto)' : '(Manual)' ?>
                                    </span>
                                </td>
                                <td><?= htmlspecialchars($aniv['enviado_por_nome'] ?: 'Sistema') ?></td>
                            <?php endif; ?>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else: ?>
                <p style="text-align: center; color: var(--color-text-secondary); padding: 40px;">
                    <i class="fas fa-inbox" style="font-size: 48px; margin-bottom: 10px;"></i><br>
                    <?= $visualizar === 'pendentes' ? 'Nenhum aniversariante pendente neste período' : 'Nenhum envio registrado neste período' ?>
                </p>
            <?php endif; ?>
        </div>
    </div>

    <!-- Modal de Configuração -->
    <div id="configModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('configModal').style.display='none'">&times;</span>
            <h2 class="section-title"><i class="fas fa-cog"></i> Configurações de Mensagens</h2>

            <form method="POST" enctype="multipart/form-data">
                <div class="campos-dinamicos">
                    <strong>Campos Dinâmicos Disponíveis:</strong><br>
                    <code>{nome}</code> - Primeiro nome |
                    <code>{sobrenome}</code> - Último nome |
                    <code>{nome_completo}</code> - Nome completo
                </div>

                <div class="filter-group">
                    <label>Assunto do Email</label>
                    <input type="text" name="mensagem_padrao_assunto" value="<?= htmlspecialchars($config_aniv['mensagem_padrao_assunto'] ?? '') ?>" required>
                </div>

                <div class="filter-group">
                    <label>Mensagem de Email (Corpo)</label>
                    <textarea name="mensagem_padrao_corpo" rows="8" required><?= htmlspecialchars($config_aniv['mensagem_padrao_corpo'] ?? '') ?></textarea>
                </div>

                <div class="filter-group">
                    <label>Mensagem de WhatsApp</label>
                    <textarea name="mensagem_whatsapp_padrao" rows="6"><?= htmlspecialchars($config_aniv['mensagem_whatsapp_padrao'] ?? '') ?></textarea>
                </div>

                <div class="filter-group">
                    <label>Imagem para Anexar (Email)</label>
                    <input type="file" name="imagem_anexo" accept="image/*">
                    <?php if (!empty($config_aniv['imagem_anexo'])): ?>
                        <small style="color: var(--color-text-secondary);">Imagem atual: <?= basename($config_aniv['imagem_anexo']) ?></small>
                    <?php endif; ?>
                </div>

                <div class="filter-group">
                    <label style="display: flex; align-items: center; gap: 10px;">
                        <input type="checkbox" name="envio_automatico_ativo" value="1" <?= $config_aniv && $config_aniv['envio_automatico_ativo'] ? 'checked' : '' ?>>
                        Ativar envio automático diário
                    </label>
                </div>

                <div style="display: flex; gap: 10px; margin-top: 20px;">
                    <button type="submit" name="salvar_config" class="btn btn-success">
                        <i class="fas fa-save"></i> Salvar Configurações
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="document.getElementById('configModal').style.display='none'">
                        Cancelar
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Modal de Envio de Mensagem -->
    <div id="envioModal" class="modal">
        <div class="modal-content">
            <span class="close" onclick="document.getElementById('envioModal').style.display='none'">&times;</span>
            <h2 class="section-title"><i class="fas fa-paper-plane"></i> Enviar Mensagem de Aniversário</h2>

            <form method="POST">
                <input type="hidden" name="convidado_id" id="envio_convidado_id">

                <div class="campos-dinamicos">
                    <strong>Enviando para:</strong> <span id="envio_nome_convidado"></span>
                </div>

                <div class="filter-group">
                    <label>Tipo de Envio</label>
                    <select name="tipo_envio" required>
                        <option value="email">Email</option>
                        <option value="whatsapp">WhatsApp</option>
                        <option value="ambos">Email + WhatsApp</option>
                    </select>
                </div>

                <div class="filter-group">
                    <label>Mensagem Personalizada</label>
                    <textarea name="mensagem_personalizada" rows="8" required id="envio_mensagem"><?= htmlspecialchars($config_aniv['mensagem_padrao_corpo'] ?? '') ?></textarea>
                    <small style="color: var(--color-text-secondary);">Use {nome}, {sobrenome}, {nome_completo}</small>
                </div>

                <div style="display: flex; gap: 10px; margin-top: 20px;">
                    <button type="submit" name="enviar_mensagem" class="btn btn-success">
                        <i class="fas fa-paper-plane"></i> Enviar Agora
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="document.getElementById('envioModal').style.display='none'">
                        Cancelar
                    </button>
                </div>
            </form>
        </div>
    </div>

    <script>
        const savedTheme = localStorage.getItem('theme') || 'dark';
        document.documentElement.setAttribute('data-theme', savedTheme);

        function abrirModalEnvio(convidadoId, nomeConvidado) {
            document.getElementById('envio_convidado_id').value = convidadoId;
            document.getElementById('envio_nome_convidado').textContent = nomeConvidado;
            document.getElementById('envioModal').style.display = 'block';
        }

        // Fechar modal ao clicar fora
        window.onclick = function(event) {
            const configModal = document.getElementById('configModal');
            const envioModal = document.getElementById('envioModal');
            if (event.target == configModal) {
                configModal.style.display = 'none';
            }
            if (event.target == envioModal) {
                envioModal.style.display = 'none';
            }
        }
    </script>

    <script src="assets/js/sidebar.js"></script>
</body>
</html>
