<?php
// admin_checkin_manual.php - Check-in manual por evento
require_once 'config.php';
require_once 'functions.php';
verificarRole(['admin']);

$sucesso = '';
$erro = '';

// Processar check-in manual
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['fazer_checkin'])) {
    $convidado_id = intval($_POST['convidado_id']);
    
    // Buscar dados do convidado
    $stmt = $pdo->prepare("SELECT * FROM convidados WHERE id = ?");
    $stmt->execute([$convidado_id]);
    $convidado = $stmt->fetch();
    
    if ($convidado) {
        if ($convidado['checkin']) {
            $erro = 'Este convidado já fez check-in!';
        } else {
            // Fazer check-in
            $admin_id = $_SESSION['user_id'];
            $stmt = $pdo->prepare("UPDATE convidados SET checkin = 1, checkin_data = NOW(), checkin_by_admin_id = ? WHERE id = ?");
            if ($stmt->execute([$admin_id, $convidado_id])) {
                // Atualizar comissões
                atualizarComissoes($pdo, $convidado['promoter_id'], $convidado['evento_id']);

                // Buscar nome do evento para notificação
                $stmt_evento = $pdo->prepare("SELECT nome FROM eventos WHERE id = ?");
                $stmt_evento->execute([$convidado['evento_id']]);
                $evento = $stmt_evento->fetch();

                // Criar notificação para o promoter (Sistema v3.0)
                if ($evento) {
                    criarNotificacaoCheckin(
                        $convidado['promoter_id'],
                        $convidado,
                        ['id' => $convidado['evento_id'], 'nome' => $evento['nome']]
                    );
                }

                $sucesso = '<i class="fas fa-check-circle"></i> Check-in realizado com sucesso para ' . htmlspecialchars($convidado['nome']) . '!';
            } else {
                $erro = 'Erro ao fazer check-in.';
            }
        }
    }
}

// Desfazer check-in
if (isset($_GET['desfazer']) && isset($_GET['evento_id'])) {
    $convidado_id = intval($_GET['desfazer']);
    $evento_id = intval($_GET['evento_id']);
    
    $stmt = $pdo->prepare("SELECT promoter_id FROM convidados WHERE id = ?");
    $stmt->execute([$convidado_id]);
    $convidado = $stmt->fetch();
    
    if ($convidado) {
        $stmt = $pdo->prepare("UPDATE convidados SET checkin = 0, checkin_data = NULL WHERE id = ?");
        if ($stmt->execute([$convidado_id])) {
            // Atualizar comissões
            atualizarComissoes($pdo, $convidado['promoter_id'], $evento_id);
            $sucesso = '<i class="fas fa-undo"></i> Check-in desfeito!';
        }
    }
}

// Buscar eventos ativos
$stmt = $pdo->query("
    SELECT e.*, 
    (SELECT COUNT(*) FROM convidados c WHERE c.evento_id = e.id) as total_convidados,
    (SELECT COUNT(*) FROM convidados c WHERE c.evento_id = e.id AND c.checkin = 1) as total_checkins
    FROM eventos e 
    WHERE e.status = 'ativo' 
    ORDER BY e.data DESC
");
$eventos = $stmt->fetchAll();

// Se um evento foi selecionado
$evento_selecionado = null;
$convidados_pendentes = [];
$convidados_confirmados = [];

if (isset($_GET['evento_id'])) {
    $evento_id = intval($_GET['evento_id']);
    
    $stmt = $pdo->prepare("SELECT * FROM eventos WHERE id = ?");
    $stmt->execute([$evento_id]);
    $evento_selecionado = $stmt->fetch();
    
    if ($evento_selecionado) {
        // Buscar convidados pendentes
        $stmt = $pdo->prepare("
            SELECT c.*, u.nome as promoter_nome 
            FROM convidados c
            INNER JOIN users u ON c.promoter_id = u.id
            WHERE c.evento_id = ? AND c.checkin = 0
            ORDER BY c.nome ASC
        ");
        $stmt->execute([$evento_id]);
        $convidados_pendentes = $stmt->fetchAll();
        
        // Buscar convidados confirmados
        $stmt = $pdo->prepare("
            SELECT c.*, u.nome as promoter_nome 
            FROM convidados c
            INNER JOIN users u ON c.promoter_id = u.id
            WHERE c.evento_id = ? AND c.checkin = 1
            ORDER BY c.checkin_data DESC
        ");
        $stmt->execute([$evento_id]);
        $convidados_confirmados = $stmt->fetchAll();
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Check-in Manual - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/dashboard-style.css">
    <link rel="stylesheet" href="assets/css/sidebar.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root[data-theme="dark"] {
            --bg-primary: #000000;
            --bg-secondary: #1a1a1a;
            --bg-card: #2a2a2a;
            --color-primary: #FFD700;
            --color-secondary: #FFA500;
            --color-text: #ffffff;
            --color-text-secondary: #999999;
            --border-color: #333333;
        }
        
        :root[data-theme="light"] {
            --bg-primary: #ffffff;
            --bg-secondary: #f5f5f5;
            --bg-card: #ffffff;
            --color-primary: #003366;
            --color-secondary: #0066cc;
            --color-text: #000000;
            --color-text-secondary: #666666;
            --border-color: #e0e0e0;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--bg-secondary);
            color: var(--color-text);
        }
.container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 30px 20px;
        }
        
        .alert {
            padding: 15px 20px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        
        .alert-success {
            background: rgba(0, 200, 81, 0.1);
            border: 2px solid #00C851;
            color: #00C851;
        }
        
        .alert-error {
            background: rgba(255, 68, 68, 0.1);
            border: 2px solid #ff4444;
            color: #ff4444;
        }
        
        .eventos-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .evento-card {
            background: var(--bg-card);
            border: 2px solid var(--border-color);
            border-radius: 15px;
            padding: 20px;
            cursor: pointer;
            transition: all 0.3s;
            text-decoration: none;
            color: var(--color-text);
        }
        
        .evento-card:hover, .evento-card.active {
            border-color: var(--color-primary);
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }
        
        .evento-nome {
            font-size: 18px;
            font-weight: bold;
            color: var(--color-primary);
            margin-bottom: 10px;
        }
        
        .evento-stats {
            display: flex;
            justify-content: space-between;
            font-size: 13px;
            color: var(--color-text-secondary);
        }
        
        .search-box {
            margin-bottom: 20px;
        }
        
        .search-input {
            width: 100%;
            padding: 15px;
            border: 2px solid var(--border-color);
            border-radius: 10px;
            background: var(--bg-card);
            color: var(--color-text);
            font-size: 16px;
        }
        
        .search-input:focus {
            outline: none;
            border-color: var(--color-primary);
        }
        
        .tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            border-bottom: 2px solid var(--border-color);
        }
        
        .tab {
            padding: 12px 20px;
            background: none;
            border: none;
            color: var(--color-text);
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            border-bottom: 3px solid transparent;
            transition: all 0.3s;
        }
        
        .tab.active {
            color: var(--color-primary);
            border-bottom-color: var(--color-primary);
        }
        
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .convidados-list {
            display: grid;
            gap: 15px;
        }
        
        .convidado-item {
            background: var(--bg-card);
            border: 2px solid var(--border-color);
            border-radius: 10px;
            padding: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: all 0.3s;
        }
        
        .convidado-item:hover {
            border-color: var(--color-primary);
        }
        
        .convidado-info {
            flex: 1;
        }
        
        .convidado-nome {
            font-size: 18px;
            font-weight: bold;
            margin-bottom: 8px;
        }
        
        .convidado-detalhes {
            font-size: 13px;
            color: var(--color-text-secondary);
            margin-bottom: 5px;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            font-size: 14px;
            font-weight: bold;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .btn-success {
            background: #00C851;
            color: white;
        }
        
        .btn-danger {
            background: #dc3545;
            color: white;
        }
        
        .btn:hover {
            opacity: 0.8;
            transform: translateY(-2px);
        }
        
        .badge {
            display: inline-block;
            padding: 4px 10px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: bold;
        }
        
        .badge-success {
            background: #00C851;
            color: white;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--color-text-secondary);
        }
        
        .empty-icon {
            font-size: 64px;
            margin-bottom: 20px;
        }
        
        @media (max-width: 768px) {
            .eventos-grid {
                grid-template-columns: 1fr;
            }
            
            .convidado-item {
                flex-direction: column;
                gap: 15px;
                text-align: center;
            }
        }
    </style>
</head>
<body>
    <?php include 'assets/sidebar.php'; ?>

    <div class="container">
        <?php if ($sucesso): ?>
            <div class="alert alert-success"><?= $sucesso ?></div>
        <?php endif; ?>
        
        <?php if ($erro): ?>
            <div class="alert alert-error"><?= $erro ?></div>
        <?php endif; ?>
        
        <h1 style="font-size: 32px; color: var(--color-primary); margin-bottom: 30px;">
            <i class="fas fa-clipboard-check"></i> Check-in Manual por Evento
        </h1>
        
        <!-- Seletor de Eventos -->
        <div class="eventos-grid">
            <?php foreach ($eventos as $evt): ?>
                <a href="?evento_id=<?= $evt['id'] ?>" class="evento-card <?= (isset($_GET['evento_id']) && $_GET['evento_id'] == $evt['id']) ? 'active' : '' ?>">
                    <div class="evento-nome"><?= htmlspecialchars($evt['nome']) ?></div>
                    <div style="font-size: 13px; color: var(--color-text-secondary); margin-bottom: 10px;">
                        <i class="fas fa-calendar-alt"></i> <?= formatarData($evt['data']) ?>
                    </div>
                    <div class="evento-stats">
                        <span>Total: <?= $evt['total_convidados'] ?></span>
                        <span><i class="fas fa-check-circle"></i> <?= $evt['total_checkins'] ?></span>
                        <span><i class="fas fa-clock"></i> <?= $evt['total_convidados'] - $evt['total_checkins'] ?></span>
                    </div>
                </a>
            <?php endforeach; ?>
        </div>
        
        <?php if ($evento_selecionado): ?>
            <!-- Caixa de Busca -->
            <div class="search-box">
                <input type="text" class="search-input" id="searchInput" placeholder="Buscar por nome, telefone ou promoter..." onkeyup="filtrarConvidados()">
            </div>
            
            <!-- Tabs -->
            <div class="tabs">
                <button class="tab active" onclick="switchTab('pendentes')">
                    <i class="fas fa-clock"></i> Pendentes (<?= count($convidados_pendentes) ?>)
                </button>
                <button class="tab" onclick="switchTab('confirmados')">
                    <i class="fas fa-check-circle"></i> Confirmados (<?= count($convidados_confirmados) ?>)
                </button>
            </div>
            
            <!-- Convidados Pendentes -->
            <div id="tab-pendentes" class="tab-content active">
                <?php if (count($convidados_pendentes) > 0): ?>
                    <div class="convidados-list" id="listaPendentes">
                        <?php foreach ($convidados_pendentes as $conv): ?>
                            <div class="convidado-item" data-nome="<?= strtolower($conv['nome']) ?>" data-promoter="<?= strtolower($conv['promoter_nome']) ?>" data-whatsapp="<?= $conv['whatsapp'] ?>">
                                <div class="convidado-info">
                                    <div class="convidado-nome"><?= htmlspecialchars($conv['nome']) ?></div>
                                    <div class="convidado-detalhes"><i class="fas fa-qrcode"></i> <?= htmlspecialchars($conv['whatsapp']) ?></div>
                                    <?php if ($conv['email']): ?>
                                        <div class="convidado-detalhes"><i class="fas fa-envelope"></i> <?= htmlspecialchars($conv['email']) ?></div>
                                    <?php endif; ?>
                                    <div class="convidado-detalhes">
                                        <i class="fas fa-user"></i> Promoter: <strong><?= htmlspecialchars($conv['promoter_nome']) ?></strong>
                                    </div>
                                </div>
                                <form method="POST" style="display: inline;">
                                    <input type="hidden" name="convidado_id" value="<?= $conv['id'] ?>">
                                    <button type="submit" name="fazer_checkin" class="btn btn-success" onclick="return confirm('Confirmar check-in de <?= htmlspecialchars($conv['nome']) ?>?')">
                                        <i class="fas fa-check-circle"></i> Fazer Check-in
                                    </button>
                                </form>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="empty-state">
                        <div class="empty-icon"><i class="fas fa-check-circle"></i></div>
                        <h3>Todos os convidados já fizeram check-in!</h3>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Convidados Confirmados -->
            <div id="tab-confirmados" class="tab-content">
                <?php if (count($convidados_confirmados) > 0): ?>
                    <div class="convidados-list" id="listaConfirmados">
                        <?php foreach ($convidados_confirmados as $conv): ?>
                            <div class="convidado-item" data-nome="<?= strtolower($conv['nome']) ?>" data-promoter="<?= strtolower($conv['promoter_nome']) ?>" data-whatsapp="<?= $conv['whatsapp'] ?>">
                                <div class="convidado-info">
                                    <div class="convidado-nome">
                                        <?= htmlspecialchars($conv['nome']) ?>
                                        <span class="badge badge-success"><i class="fas fa-check-circle"></i> Confirmado</span>
                                    </div>
                                    <div class="convidado-detalhes"><i class="fas fa-qrcode"></i> <?= htmlspecialchars($conv['whatsapp']) ?></div>
                                    <div class="convidado-detalhes">
                                        <i class="fas fa-user"></i> Promoter: <strong><?= htmlspecialchars($conv['promoter_nome']) ?></strong>
                                    </div>
                                    <div class="convidado-detalhes">
                                        <i class="fas fa-clock"></i> Check-in: <?= formatarDataHora($conv['checkin_data']) ?>
                                    </div>
                                </div>
                                <a href="?evento_id=<?= $evento_selecionado['id'] ?>&desfazer=<?= $conv['id'] ?>" class="btn btn-danger" onclick="return confirm('Desfazer check-in de <?= htmlspecialchars($conv['nome']) ?>?')">
                                    <i class="fas fa-undo"></i> Desfazer
                                </a>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="empty-state">
                        <div class="empty-icon"><i class="fas fa-clock"></i></div>
                        <h3>Nenhum check-in realizado ainda</h3>
                    </div>
                <?php endif; ?>
            </div>
        <?php elseif (count($eventos) === 0): ?>
            <div class="empty-state">
                <div class="empty-icon"><i class="fas fa-theater-masks"></i></div>
                <h3>Nenhum evento ativo</h3>
                <p>Crie um evento para começar.</p>
            </div>
        <?php endif; ?>
    </div>
    
    <script>
        const savedTheme = localStorage.getItem('theme') || 'dark';
        document.documentElement.setAttribute('data-theme', savedTheme);
        
        function switchTab(tabName) {
            document.querySelectorAll('.tab').forEach(t => t.classList.remove('active'));
            document.querySelectorAll('.tab-content').forEach(c => c.classList.remove('active'));
            
            event.target.classList.add('active');
            document.getElementById('tab-' + tabName).classList.add('active');
        }
        
        function filtrarConvidados() {
            const searchTerm = document.getElementById('searchInput').value.toLowerCase();
            const items = document.querySelectorAll('.convidado-item');
            
            items.forEach(item => {
                const nome = item.getAttribute('data-nome');
                const promoter = item.getAttribute('data-promoter');
                const whatsapp = item.getAttribute('data-whatsapp');
                
                if (nome.includes(searchTerm) || promoter.includes(searchTerm) || whatsapp.includes(searchTerm)) {
                    item.style.display = 'flex';
                } else {
                    item.style.display = 'none';
                }
            });
        }
    </script>
    </div><!-- /.main-content -->
    <script src="assets/js/sidebar.js"></script>
</body>
</html>