<?php
// api/mensagens_api.php - API de Mensagens v3.0
header('Content-Type: application/json');
require_once '../config.php';

// Verificar se está logado
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Não autorizado']);
    exit;
}

$user_id = $_SESSION['user_id'];
$action = $_GET['action'] ?? '';

try {
    switch ($action) {
        case 'contar':
            // Contar mensagens não lidas
            $stmt = $pdo->prepare("
                SELECT
                    COUNT(*) as total,
                    COALESCE(SUM(CASE WHEN md.lida = 0 THEN 1 ELSE 0 END), 0) as nao_lidas,
                    COALESCE(SUM(CASE WHEN md.lida = 1 THEN 1 ELSE 0 END), 0) as lidas,
                    COALESCE(SUM(CASE WHEN md.arquivada = 1 THEN 1 ELSE 0 END), 0) as arquivadas
                FROM mensagens_destinatarios md
                WHERE md.destinatario_id = ?
            ");
            $stmt->execute([$user_id]);
            $contadores = $stmt->fetch(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'nao_lidas' => (int)$contadores['nao_lidas'],
                'contadores' => $contadores
            ]);
            break;

        case 'listar':
            // Listar mensagens
            $filtro = $_GET['filtro'] ?? 'todas';
            $limit = min((int)($_GET['limit'] ?? 20), 100);
            $offset = max((int)($_GET['offset'] ?? 0), 0);

            $where_filtro = "md.arquivada = 0";

            switch ($filtro) {
                case 'nao_lidas':
                    $where_filtro = "md.lida = 0";
                    break;
                case 'lidas':
                    $where_filtro = "md.lida = 1";
                    break;
                case 'arquivadas':
                    $where_filtro = "md.arquivada = 1";
                    break;
            }

            $stmt = $pdo->prepare("
                SELECT m.*, md.lida, md.lida_em, md.arquivada
                FROM mensagens m
                INNER JOIN mensagens_destinatarios md ON m.id = md.mensagem_id
                WHERE md.destinatario_id = ? AND $where_filtro
                ORDER BY m.created_at DESC
                LIMIT $limit OFFSET $offset
            ");
            $stmt->execute([$user_id]);
            $mensagens = $stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'mensagens' => $mensagens,
                'total' => count($mensagens)
            ]);
            break;

        case 'marcar_lida':
            // Marcar mensagem como lida
            $mensagem_id = (int)($_POST['id'] ?? 0);

            if (!$mensagem_id) {
                throw new Exception('ID da mensagem não fornecido');
            }

            $stmt = $pdo->prepare("
                UPDATE mensagens_destinatarios
                SET lida = 1, lida_em = NOW()
                WHERE mensagem_id = ? AND destinatario_id = ?
            ");
            $stmt->execute([$mensagem_id, $user_id]);

            // Atualizar contador
            $stmt = $pdo->prepare("
                UPDATE users
                SET mensagens_nao_lidas = GREATEST(mensagens_nao_lidas - 1, 0)
                WHERE id = ?
            ");
            $stmt->execute([$user_id]);

            echo json_encode([
                'success' => true,
                'message' => 'Mensagem marcada como lida'
            ]);
            break;

        case 'arquivar':
            // Arquivar mensagem
            $mensagem_id = (int)($_POST['id'] ?? 0);

            if (!$mensagem_id) {
                throw new Exception('ID da mensagem não fornecido');
            }

            $stmt = $pdo->prepare("
                UPDATE mensagens_destinatarios
                SET arquivada = 1
                WHERE mensagem_id = ? AND destinatario_id = ?
            ");
            $stmt->execute([$mensagem_id, $user_id]);

            echo json_encode([
                'success' => true,
                'message' => 'Mensagem arquivada'
            ]);
            break;

        case 'ultimas':
            // Buscar últimas mensagens não lidas
            $limit = min((int)($_GET['limit'] ?? 5), 20);

            $stmt = $pdo->prepare("
                SELECT m.*, md.lida, md.lida_em
                FROM mensagens m
                INNER JOIN mensagens_destinatarios md ON m.id = md.mensagem_id
                WHERE md.destinatario_id = ? AND md.lida = 0 AND md.arquivada = 0
                ORDER BY m.created_at DESC
                LIMIT $limit
            ");
            $stmt->execute([$user_id]);
            $mensagens = $stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'mensagens' => $mensagens
            ]);
            break;

        default:
            throw new Exception('Ação não reconhecida');
    }

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
