<?php
// configuracoes.php
require_once 'config.php';

// Verificar se é admin
if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$sucesso = '';
$erro = '';

// Buscar dados do usuário para sidebar
$stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$user_id]);
$usuario = $stmt->fetch();

// Buscar configurações atuais
$stmt = $pdo->query("SELECT * FROM configuracoes WHERE id = 1");
$config = $stmt->fetch();

// Se não existir, criar registro padrão
if (!$config) {
    $pdo->exec("INSERT INTO configuracoes (id, nome_sistema, logo_sistema, banner_principal, cor_primaria, cor_secundaria, mostrar_whatsapp, whatsapp_numero, mostrar_instagram, instagram_usuario) VALUES (1, 'Sistema Promoters', '', '', '#FFD700', '#FFA500', 1, '', 1, '')");
    $stmt = $pdo->query("SELECT * FROM configuracoes WHERE id = 1");
    $config = $stmt->fetch();
}

// Processar formulário
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Usar htmlspecialchars em vez de FILTER_SANITIZE_STRING (deprecated no PHP 8.1+)
    $nome_sistema = isset($_POST['nome_sistema']) ? htmlspecialchars(trim($_POST['nome_sistema']), ENT_QUOTES, 'UTF-8') : '';
    $cor_primaria = isset($_POST['cor_primaria']) ? htmlspecialchars(trim($_POST['cor_primaria']), ENT_QUOTES, 'UTF-8') : '#FFD700';
    $cor_secundaria = isset($_POST['cor_secundaria']) ? htmlspecialchars(trim($_POST['cor_secundaria']), ENT_QUOTES, 'UTF-8') : '#FFA500';
    $mostrar_whatsapp = isset($_POST['mostrar_whatsapp']) ? 1 : 0;
    $whatsapp_numero = isset($_POST['whatsapp_numero']) ? htmlspecialchars(trim($_POST['whatsapp_numero']), ENT_QUOTES, 'UTF-8') : '';
    $mostrar_instagram = isset($_POST['mostrar_instagram']) ? 1 : 0;
    $instagram_usuario = isset($_POST['instagram_usuario']) ? htmlspecialchars(trim($_POST['instagram_usuario']), ENT_QUOTES, 'UTF-8') : '';

    // Configurações de Email
    $email_remetente = isset($_POST['email_remetente']) ? htmlspecialchars(trim($_POST['email_remetente']), ENT_QUOTES, 'UTF-8') : 'noreply@seudominio.com.br';
    $email_nome_remetente = isset($_POST['email_nome_remetente']) ? htmlspecialchars(trim($_POST['email_nome_remetente']), ENT_QUOTES, 'UTF-8') : 'Sistema Promotaal';
    
    $logo_sistema = $config['logo_sistema'];
    $banner_principal = $config['banner_principal'];

    try {
        // Upload do logo
        if (isset($_FILES['logo_sistema']) && $_FILES['logo_sistema']['error'] == 0) {
            $allowed = ['jpg', 'jpeg', 'png', 'gif', 'svg'];
            $filename = $_FILES['logo_sistema']['name'];
            $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

            if (in_array($ext, $allowed)) {
                $novo_nome = 'logo_' . time() . '.' . $ext;
                $destino = 'uploads/sistema/' . $novo_nome;

                if (!is_dir('uploads/sistema')) {
                    mkdir('uploads/sistema', 0777, true);
                }

                if (move_uploaded_file($_FILES['logo_sistema']['tmp_name'], $destino)) {
                    // Deletar logo antigo se existir
                    if (!empty($logo_sistema) && file_exists($logo_sistema) && $logo_sistema !== $destino) {
                        @unlink($logo_sistema);
                    }
                    $logo_sistema = $destino;
                } else {
                    $erro = 'Erro ao fazer upload do logo. Verifique as permissões da pasta uploads/sistema.';
                }
            } else {
                $erro = 'Formato de arquivo inválido para logo. Use JPG, PNG, GIF ou SVG.';
            }
        }

        // Upload do banner
        if (isset($_FILES['banner_principal']) && $_FILES['banner_principal']['error'] == 0 && empty($erro)) {
            $allowed = ['jpg', 'jpeg', 'png', 'gif'];
            $filename = $_FILES['banner_principal']['name'];
            $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));

            if (in_array($ext, $allowed)) {
                $novo_nome = 'banner_' . time() . '.' . $ext;
                $destino = 'uploads/sistema/' . $novo_nome;

                if (!is_dir('uploads/sistema')) {
                    mkdir('uploads/sistema', 0777, true);
                }

                if (move_uploaded_file($_FILES['banner_principal']['tmp_name'], $destino)) {
                    // Deletar banner antigo se existir
                    if (!empty($banner_principal) && file_exists($banner_principal) && $banner_principal !== $destino) {
                        @unlink($banner_principal);
                    }
                    $banner_principal = $destino;
                } else {
                    $erro = 'Erro ao fazer upload do banner. Verifique as permissões da pasta uploads/sistema.';
                }
            } else {
                $erro = 'Formato de arquivo inválido para banner. Use JPG, PNG ou GIF.';
            }
        }
    } catch (Exception $e) {
        $erro = 'Erro ao processar uploads: ' . $e->getMessage();
        error_log('Erro em configuracoes.php upload: ' . $e->getMessage());
    }
    
    // Atualizar no banco (apenas se não houver erros de upload)
    if (empty($erro)) {
        try {
            $stmt = $pdo->prepare("UPDATE configuracoes SET nome_sistema = ?, logo_sistema = ?, banner_principal = ?, cor_primaria = ?, cor_secundaria = ?, mostrar_whatsapp = ?, whatsapp_numero = ?, mostrar_instagram = ?, instagram_usuario = ?, email_remetente = ?, email_nome_remetente = ? WHERE id = 1");

            if ($stmt->execute([$nome_sistema, $logo_sistema, $banner_principal, $cor_primaria, $cor_secundaria, $mostrar_whatsapp, $whatsapp_numero, $mostrar_instagram, $instagram_usuario, $email_remetente, $email_nome_remetente])) {
                $sucesso = 'Configurações salvas com sucesso!';
                // Recarregar configurações
                $stmt = $pdo->query("SELECT * FROM configuracoes WHERE id = 1");
                $config = $stmt->fetch();
            } else {
                $erro = 'Erro ao salvar configurações no banco de dados.';
            }
        } catch (PDOException $e) {
            $erro = 'Erro ao salvar: ' . $e->getMessage();
            error_log('Erro em configuracoes.php database: ' . $e->getMessage());
        }
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Configurações - <?= htmlspecialchars($config['nome_sistema']) ?></title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/dashboard-style.css">
    <link rel="stylesheet" href="assets/css/sidebar.css">
    <style>
        .config-container {
            max-width: 1100px;
            margin: 0 auto;
            padding: 30px 20px;
        }

        .header {
            margin-bottom: 40px;
        }

        .header-title {
            font-size: 32px;
            font-weight: 700;
            color: #FFD700;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .header-subtitle {
            color: #999999;
            font-size: 16px;
        }

        .config-sections {
            display: flex;
            flex-direction: column;
            gap: 30px;
        }

        .config-section {
            background: #1a1a1a;
            border-radius: 15px;
            padding: 30px;
            border: 2px solid #2a2a2a;
            transition: all 0.3s ease;
        }

        .config-section:hover {
            border-color: #FFD700;
            box-shadow: 0 5px 20px rgba(255, 215, 0, 0.1);
        }

        .config-section:nth-child(1) { animation-delay: 0.1s; }
        .config-section:nth-child(2) { animation-delay: 0.2s; }
        .config-section:nth-child(3) { animation-delay: 0.3s; }
        .config-section:nth-child(4) { animation-delay: 0.4s; }

        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .section-header {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 25px;
            padding-bottom: 20px;
            border-bottom: 2px solid #2a2a2a;
        }

        .section-icon {
            width: 50px;
            height: 50px;
            background: linear-gradient(135deg, #FFD700 0%, #FFA500 100%);
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: #000;
        }

        .section-title {
            font-size: 22px;
            font-weight: 700;
            color: #FFFFFF;
        }

        .preview-box {
            background: #0a0a0a;
            border-radius: 10px;
            padding: 20px;
            margin-top: 15px;
            border: 2px dashed #2a2a2a;
            text-align: center;
        }

        .preview-image {
            max-width: 100%;
            max-height: 200px;
            border-radius: 10px;
            margin-top: 15px;
        }

        .color-picker-wrapper {
            display: flex;
            align-items: center;
            gap: 15px;
        }

        .color-preview {
            width: 60px;
            height: 60px;
            border-radius: 10px;
            border: 3px solid #2a2a2a;
            cursor: pointer;
            transition: all 0.3s;
        }

        .color-preview:hover {
            transform: scale(1.1);
            border-color: #FFD700;
        }

        input[type="color"] {
            opacity: 0;
            width: 0;
            height: 0;
            position: absolute;
        }

        .switch-wrapper {
            display: flex;
            align-items: center;
            justify-content: space-between;
            padding: 18px;
            background: #0a0a0a;
            border-radius: 10px;
            margin-bottom: 15px;
        }

        .switch-label {
            display: flex;
            align-items: center;
            gap: 12px;
            font-weight: 600;
            color: #FFFFFF;
        }

        .switch {
            position: relative;
            width: 60px;
            height: 32px;
            background: #2a2a2a;
            border-radius: 50px;
            cursor: pointer;
            transition: background 0.3s;
        }

        .switch::before {
            content: '';
            position: absolute;
            width: 24px;
            height: 24px;
            border-radius: 50%;
            background: #666666;
            top: 4px;
            left: 4px;
            transition: all 0.3s;
        }

        input[type="checkbox"]:checked + .switch {
            background: linear-gradient(135deg, #FFD700 0%, #FFA500 100%);
        }

        input[type="checkbox"]:checked + .switch::before {
            left: 32px;
            background: #FFFFFF;
        }

        input[type="checkbox"] {
            display: none;
        }

        .actions {
            display: flex;
            gap: 15px;
            margin-top: 40px;
        }

        .file-input-wrapper {
            position: relative;
            overflow: hidden;
            display: inline-block;
            width: 100%;
        }

        .file-input-label {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 12px;
            padding: 18px;
            background: #0a0a0a;
            border: 2px dashed #2a2a2a;
            border-radius: 10px;
            cursor: pointer;
            transition: all 0.3s;
            font-weight: 600;
            color: #999999;
        }

        .file-input-label:hover {
            border-color: #FFD700;
            background: rgba(255, 215, 0, 0.05);
            color: #FFD700;
        }

        .file-input-wrapper input[type="file"] {
            position: absolute;
            opacity: 0;
            cursor: pointer;
        }

        /* Estilos dos Botões Melhorados */
        .btn-primary, .btn-secondary {
            display: inline-flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
            padding: 16px 32px;
            font-size: 16px;
            font-weight: 700;
            border: none;
            border-radius: 12px;
            cursor: pointer;
            transition: all 0.3s ease;
            text-decoration: none;
            position: relative;
            overflow: hidden;
        }

        .btn-primary {
            background: linear-gradient(135deg, #FFD700 0%, #FFA500 100%);
            color: #000000;
            box-shadow: 0 8px 25px rgba(255, 215, 0, 0.3);
        }

        .btn-primary:hover {
            transform: translateY(-3px);
            box-shadow: 0 12px 35px rgba(255, 215, 0, 0.5);
            background: linear-gradient(135deg, #FFA500 0%, #FFD700 100%);
        }

        .btn-primary:active {
            transform: translateY(-1px);
            box-shadow: 0 5px 15px rgba(255, 215, 0, 0.3);
        }

        .btn-primary::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.3);
            transform: translate(-50%, -50%);
            transition: width 0.6s, height 0.6s;
        }

        .btn-primary:hover::before {
            width: 300px;
            height: 300px;
        }

        .btn-primary i {
            font-size: 18px;
            z-index: 1;
        }

        .btn-primary span {
            z-index: 1;
        }

        .btn-secondary {
            background: #2a2a2a;
            color: #FFFFFF;
            border: 2px solid #3a3a3a;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.3);
        }

        .btn-secondary:hover {
            background: #3a3a3a;
            border-color: #FFD700;
            color: #FFD700;
            transform: translateY(-2px);
            box-shadow: 0 8px 20px rgba(255, 215, 0, 0.2);
        }

        .btn-secondary:active {
            transform: translateY(0);
        }

        .btn-secondary i {
            font-size: 16px;
        }

        @media (max-width: 768px) {
            .config-container {
                padding: 20px 15px;
            }

            .header-title {
                font-size: 24px;
            }

            .config-section {
                padding: 20px;
            }

            .actions {
                flex-direction: column;
            }

            .btn-primary, .btn-secondary {
                width: 100%;
                padding: 14px 24px;
                font-size: 15px;
            }

            .section-title {
                font-size: 18px;
            }
        }
    </style>
</head>
<body>
    <?php include 'assets/sidebar.php'; ?>

    <div class="config-container">
        <div class="header">
            <h1 class="header-title"><i class="fas fa-cog"></i> Configurações do Sistema</h1>
            <p class="header-subtitle">Personalize a aparência e funcionalidades do sistema</p>
        </div>

        <?php if ($sucesso): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?= htmlspecialchars($sucesso) ?>
            </div>
        <?php endif; ?>

        <?php if ($erro): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-triangle"></i>
                <?= htmlspecialchars($erro) ?>
            </div>
        <?php endif; ?>

        <form method="POST" enctype="multipart/form-data">
            <div class="config-sections">
                
                <!-- Informações Básicas -->
                <div class="config-section">
                    <div class="section-header">
                        <div class="section-icon"><i class="fas fa-building"></i></div>
                        <h2 class="section-title">Informações Básicas</h2>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Nome do Sistema</label>
                        <input type="text" name="nome_sistema" class="form-input" 
                               value="<?= htmlspecialchars($config['nome_sistema']) ?>" required>
                    </div>
                </div>

                <!-- Logo e Banner -->
                <div class="config-section">
                    <div class="section-header">
                        <div class="section-icon"><i class="fas fa-palette"></i></div>
                        <h2 class="section-title">Identidade Visual</h2>
                    </div>

                    <div class="grid-2">
                        <div class="form-group">
                            <label class="form-label">Logo do Sistema</label>
                            <div class="file-input-wrapper">
                                <label class="file-input-label" for="logo_sistema">
                                    <i class="fas fa-folder-open"></i>
                                    Escolher Logo
                                </label>
                                <input type="file" id="logo_sistema" name="logo_sistema" accept="image/*" onchange="previewImage(event, 'logo-preview')">
                            </div>
                            <?php if (!empty($config['logo_sistema'])): ?>
                                <div class="preview-box">
                                    <p style="color: var(--text-secondary); font-size: 13px; margin-bottom: 8px;">Logo Atual</p>
                                    <img src="<?= htmlspecialchars($config['logo_sistema']) ?>" class="preview-image" id="logo-preview">
                                </div>
                            <?php endif; ?>
                        </div>

                        <div class="form-group">
                            <label class="form-label">Banner Principal</label>
                            <div class="file-input-wrapper">
                                <label class="file-input-label" for="banner_principal">
                                    <i class="fas fa-image"></i>
                                    Escolher Banner
                                </label>
                                <input type="file" id="banner_principal" name="banner_principal" accept="image/*" onchange="previewImage(event, 'banner-preview')">
                            </div>
                            <?php if (!empty($config['banner_principal'])): ?>
                                <div class="preview-box">
                                    <p style="color: var(--text-secondary); font-size: 13px; margin-bottom: 8px;">Banner Atual</p>
                                    <img src="<?= htmlspecialchars($config['banner_principal']) ?>" class="preview-image" id="banner-preview">
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <!-- Cores do Sistema -->
                <div class="config-section">
                    <div class="section-header">
                        <div class="section-icon"><i class="fas fa-palette"></i></div>
                        <h2 class="section-title">Paleta de Cores</h2>
                    </div>

                    <div class="grid-2">
                        <div class="form-group">
                            <label class="form-label">Cor Primária</label>
                            <div class="color-picker-wrapper">
                                <label for="cor_primaria" class="color-preview" 
                                       style="background-color: <?= htmlspecialchars($config['cor_primaria']) ?>;"
                                       onclick="document.getElementById('cor_primaria').click()">
                                </label>
                                <input type="color" id="cor_primaria" name="cor_primaria" 
                                       value="<?= htmlspecialchars($config['cor_primaria']) ?>"
                                       onchange="this.previousElementSibling.style.backgroundColor = this.value">
                                <input type="text" class="form-input" 
                                       value="<?= htmlspecialchars($config['cor_primaria']) ?>" 
                                       readonly style="flex: 1;">
                            </div>
                        </div>

                        <div class="form-group">
                            <label class="form-label">Cor Secundária</label>
                            <div class="color-picker-wrapper">
                                <label for="cor_secundaria" class="color-preview" 
                                       style="background-color: <?= htmlspecialchars($config['cor_secundaria']) ?>;"
                                       onclick="document.getElementById('cor_secundaria').click()">
                                </label>
                                <input type="color" id="cor_secundaria" name="cor_secundaria" 
                                       value="<?= htmlspecialchars($config['cor_secundaria']) ?>"
                                       onchange="this.previousElementSibling.style.backgroundColor = this.value">
                                <input type="text" class="form-input" 
                                       value="<?= htmlspecialchars($config['cor_secundaria']) ?>" 
                                       readonly style="flex: 1;">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Redes Sociais -->
                <div class="config-section">
                    <div class="section-header">
                        <div class="section-icon"><i class="fas fa-mobile-alt"></i></div>
                        <h2 class="section-title">Redes Sociais</h2>
                    </div>

                    <div class="switch-wrapper">
                        <label class="switch-label">
                            <i class="fab fa-whatsapp"></i>
                            Mostrar WhatsApp
                        </label>
                        <input type="checkbox" id="mostrar_whatsapp" name="mostrar_whatsapp"
                               <?= $config['mostrar_whatsapp'] ? 'checked' : '' ?>>
                        <label for="mostrar_whatsapp" class="switch"></label>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Número do WhatsApp</label>
                        <input type="text" name="whatsapp_numero" class="form-input"
                               placeholder="(11) 99999-9999"
                               value="<?= htmlspecialchars($config['whatsapp_numero']) ?>">
                    </div>

                    <div class="switch-wrapper">
                        <label class="switch-label">
                            <i class="fab fa-instagram"></i>
                            Mostrar Instagram
                        </label>
                        <input type="checkbox" id="mostrar_instagram" name="mostrar_instagram"
                               <?= $config['mostrar_instagram'] ? 'checked' : '' ?>>
                        <label for="mostrar_instagram" class="switch"></label>
                    </div>

                    <div class="form-group">
                        <label class="form-label">Usuário do Instagram</label>
                        <input type="text" name="instagram_usuario" class="form-input"
                               placeholder="@usuario"
                               value="<?= htmlspecialchars($config['instagram_usuario']) ?>">
                    </div>
                </div>

                <!-- Configurações de Email -->
                <div class="config-section">
                    <div class="section-header">
                        <div class="section-icon"><i class="fas fa-envelope"></i></div>
                        <h2 class="section-title">Configurações de Email</h2>
                    </div>

                    <div style="background: #0a0a0a; padding: 15px; border-radius: 10px; margin-bottom: 25px; border-left: 4px solid #FFD700;">
                        <p style="color: #FFD700; font-size: 14px; margin: 0; display: flex; align-items: center; gap: 10px;">
                            <i class="fas fa-info-circle"></i>
                            <strong>Envio Nativo (PHP mail())</strong>
                        </p>
                        <p style="color: #999; font-size: 13px; margin: 8px 0 0 0; line-height: 1.6;">
                            Esta versão usa o envio nativo do PHP através do servidor cPanel. Configure o email e nome do remetente abaixo.
                            Em versões futuras, será implementado suporte a SMTP para maior controle.
                        </p>
                    </div>

                    <div class="form-group">
                        <label class="form-label">
                            <i class="fas fa-at"></i> Email Remetente
                        </label>
                        <input type="email" name="email_remetente" class="form-input"
                               placeholder="noreply@seudominio.com.br"
                               value="<?= isset($config['email_remetente']) ? htmlspecialchars($config['email_remetente']) : 'noreply@seudominio.com.br' ?>"
                               required>
                        <small style="color: #999; font-size: 12px; display: block; margin-top: 8px;">
                            Email que aparecerá como remetente nos emails enviados pelo sistema
                        </small>
                    </div>

                    <div class="form-group">
                        <label class="form-label">
                            <i class="fas fa-user"></i> Nome do Remetente
                        </label>
                        <input type="text" name="email_nome_remetente" class="form-input"
                               placeholder="Sistema Promotaal"
                               value="<?= isset($config['email_nome_remetente']) ? htmlspecialchars($config['email_nome_remetente']) : 'Sistema Promotaal' ?>"
                               required>
                        <small style="color: #999; font-size: 12px; display: block; margin-top: 8px;">
                            Nome que aparecerá como remetente nos emails enviados pelo sistema
                        </small>
                    </div>
                </div>

            </div>

            <div class="actions">
                <button type="submit" class="btn-primary" style="flex: 1;">
                    <i class="fas fa-save"></i>
                    <span>Salvar Configurações</span>
                </button>
                <a href="dashboard.php" class="btn-secondary">
                    <i class="fas fa-arrow-left"></i>
                    <span>Voltar ao Dashboard</span>
                </a>
            </div>
        </form>
    </div>

    <script>
        function previewImage(event, previewId) {
            const file = event.target.files[0];
            const preview = document.getElementById(previewId);
            
            if (file && preview) {
                const reader = new FileReader();
                reader.onload = function(e) {
                    preview.src = e.target.result;
                    
                    // Criar preview box se não existir
                    if (!preview.closest('.preview-box')) {
                        const previewBox = document.createElement('div');
                        previewBox.className = 'preview-box';
                        previewBox.innerHTML = '<p style="color: var(--text-secondary); font-size: 13px; margin-bottom: 8px;">Pré-visualização</p>';
                        previewBox.appendChild(preview);
                        event.target.closest('.form-group').appendChild(previewBox);
                    }
                }
                reader.readAsDataURL(file);
            }
        }
    </script>
    </div><!-- /.main-content -->
    <script src="assets/js/sidebar.js"></script>
</body>
</html>