<?php
/**
 * cron_aniversarios.php
 * Script para envio automático de mensagens de aniversário
 *
 * Execute este script diariamente via cron job:
 * 0 9 * * * /usr/bin/php /caminho/para/cron_aniversarios.php
 *
 * Ou adicione no cPanel: Cron Jobs > 0 9 * * * php /home/usuario/public_html/cron_aniversarios.php
 */

require_once __DIR__ . '/config.php';

// Log de execução
$log_file = __DIR__ . '/logs/aniversarios_' . date('Y-m') . '.log';
$log_dir = dirname($log_file);
if (!file_exists($log_dir)) {
    mkdir($log_dir, 0755, true);
}

function logMsg($mensagem) {
    global $log_file;
    $timestamp = date('Y-m-d H:i:s');
    file_put_contents($log_file, "[$timestamp] $mensagem\n", FILE_APPEND);
    echo "[$timestamp] $mensagem\n";
}

logMsg("=== INICIANDO ENVIO AUTOMÁTICO DE ANIVERSÁRIOS ===");

try {
    // Buscar configurações
    $stmt = $pdo->query("SELECT * FROM configuracoes_aniversarios LIMIT 1");
    $config = $stmt->fetch();

    if (!$config || !$config['envio_automatico_ativo']) {
        logMsg("Envio automático está desativado. Abortando.");
        exit;
    }

    $hoje_dia = intval(date('d'));
    $hoje_mes = intval(date('m'));
    $ano_atual = intval(date('Y'));

    logMsg("Buscando aniversariantes de hoje: dia $hoje_dia, mês $hoje_mes");

    // Buscar aniversariantes de hoje que ainda não receberam mensagem este ano
    $stmt = $pdo->prepare("
        SELECT c.*
        FROM convidados c
        LEFT JOIN aniversarios_enviados ae ON c.id = ae.convidado_id AND ae.ano = ?
        WHERE c.data_nascimento IS NOT NULL
          AND DAY(c.data_nascimento) = ?
          AND MONTH(c.data_nascimento) = ?
          AND ae.id IS NULL
    ");
    $stmt->execute([$ano_atual, $hoje_dia, $hoje_mes]);
    $aniversariantes = $stmt->fetchAll();

    $total = count($aniversariantes);
    logMsg("Encontrados $total aniversariantes para enviar hoje");

    if ($total === 0) {
        logMsg("Nenhum aniversariante para processar. Finalizando.");
        exit;
    }

    $enviados = 0;
    $erros = 0;

    foreach ($aniversariantes as $convidado) {
        try {
            logMsg("Processando: {$convidado['nome']} (ID: {$convidado['id']})");

            // Preparar campos dinâmicos
            $nome_partes = explode(' ', trim($convidado['nome']));
            $primeiro_nome = $nome_partes[0];
            $sobrenome = count($nome_partes) > 1 ? end($nome_partes) : '';

            // Substituir campos na mensagem
            $assunto = str_replace(
                ['{nome}', '{sobrenome}', '{nome_completo}'],
                [$primeiro_nome, $sobrenome, $convidado['nome']],
                $config['mensagem_padrao_assunto']
            );

            $corpo = str_replace(
                ['{nome}', '{sobrenome}', '{nome_completo}'],
                [$primeiro_nome, $sobrenome, $convidado['nome']],
                $config['mensagem_padrao_corpo']
            );

            $enviou_email = false;

            // Verificar se tem email válido
            if (!empty($convidado['email']) && filter_var($convidado['email'], FILTER_VALIDATE_EMAIL)) {
                // Preparar headers do email - usa constantes do config.php
                $headers = "MIME-Version: 1.0\r\n";
                $headers .= "Content-Type: text/html; charset=UTF-8\r\n";
                $headers .= "From: " . EMAIL_NOME_REMETENTE . " <" . EMAIL_REMETENTE . ">\r\n";

                // Converter quebras de linha para HTML
                $corpo_html = nl2br(htmlspecialchars($corpo));

                // HTML do email
                $html_email = "
                <!DOCTYPE html>
                <html>
                <head>
                    <meta charset='UTF-8'>
                    <style>
                        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                        .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                        .header { background: linear-gradient(135deg, #FFD700 0%, #FFA500 100%); color: white; padding: 30px; text-align: center; border-radius: 10px 10px 0 0; }
                        .content { background: #fff; padding: 30px; border: 1px solid #ddd; }
                        .footer { background: #f5f5f5; padding: 20px; text-align: center; font-size: 12px; color: #666; border-radius: 0 0 10px 10px; }
                        .emoji { font-size: 48px; }
                    </style>
                </head>
                <body>
                    <div class='container'>
                        <div class='header'>
                            <div class='emoji'>🎂🎉🎈</div>
                            <h1>Feliz Aniversário!</h1>
                        </div>
                        <div class='content'>
                            $corpo_html
                        </div>
                        <div class='footer'>
                            Esta é uma mensagem automática. Por favor, não responda.
                        </div>
                    </div>
                </body>
                </html>
                ";

                // Enviar email
                if (mail($convidado['email'], $assunto, $html_email, $headers)) {
                    $enviou_email = true;
                    logMsg("  ✓ Email enviado para: {$convidado['email']}");
                } else {
                    logMsg("  ✗ Falha ao enviar email para: {$convidado['email']}");
                }
            } else {
                logMsg("  - Email inválido ou não informado");
            }

            if ($enviou_email) {
                // Registrar no histórico
                $stmt = $pdo->prepare("
                    INSERT INTO aniversarios_enviados (convidado_id, ano, tipo_envio, enviado_por, mensagem_enviada, enviado_automatico)
                    VALUES (?, ?, 'email', 1, ?, 1)
                ");
                $stmt->execute([$convidado['id'], $ano_atual, $corpo]);

                // Criar notificação para admins
                $stmt_admins = $pdo->query("SELECT id FROM users WHERE role = 'admin'");
                $admins = $stmt_admins->fetchAll();

                foreach ($admins as $admin) {
                    criarNotificacao([
                        'user_id' => $admin['id'],
                        'tipo' => 'sistema',
                        'titulo' => 'Aniversário enviado automaticamente',
                        'mensagem' => "Mensagem de aniversário enviada para {$convidado['nome']}",
                        'referencia_tipo' => 'aniversario',
                        'referencia_id' => $convidado['id']
                    ]);
                }

                $enviados++;
                logMsg("  ✓ Registrado no histórico");
            } else {
                $erros++;
            }

        } catch (Exception $e) {
            logMsg("  ✗ ERRO ao processar {$convidado['nome']}: " . $e->getMessage());
            $erros++;
        }
    }

    logMsg("=== PROCESSAMENTO CONCLUÍDO ===");
    logMsg("Total processados: $total | Enviados: $enviados | Erros: $erros");

} catch (Exception $e) {
    logMsg("ERRO FATAL: " . $e->getMessage());
    logMsg($e->getTraceAsString());
}

logMsg("=== FIM DA EXECUÇÃO ===\n");
