<?php
// functions.php - Funções de cálculo de comissões e envio de email
// ✅ CORRIGIDO: Removida função formatarData() duplicada

/**
 * Atualizar comissões de um promoter para um evento específico
 */
function atualizarComissoes($pdo, $promoter_id, $evento_id) {
    try {
        // Buscar informações do evento
        $stmt = $pdo->prepare("SELECT valor_comissao, minimo_lista FROM eventos WHERE id = ?");
        $stmt->execute([$evento_id]);
        $evento = $stmt->fetch();
        
        if (!$evento) {
            error_log("Evento não encontrado: " . $evento_id);
            return false;
        }
        
        // Contar convidados total e check-ins
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(*) as total,
                SUM(CASE WHEN checkin = 1 THEN 1 ELSE 0 END) as checkins
            FROM convidados 
            WHERE promoter_id = ? AND evento_id = ?
        ");
        $stmt->execute([$promoter_id, $evento_id]);
        $stats = $stmt->fetch();
        
        $total_convidados = intval($stats['total']);
        $total_checkins = intval($stats['checkins']);
        
        error_log("Promoter ID: $promoter_id | Evento ID: $evento_id | Total: $total_convidados | Check-ins: $total_checkins | Mínimo: {$evento['minimo_lista']}");
        
        // Verificar se atingiu o mínimo para receber comissão
        if ($total_convidados < $evento['minimo_lista']) {
            error_log("Não atingiu mínimo. Removendo comissão se existir.");
            // Não atingiu o mínimo - remover comissão se existir
            $stmt = $pdo->prepare("DELETE FROM comissoes WHERE user_id = ? AND evento_id = ?");
            $stmt->execute([$promoter_id, $evento_id]);
            return true;
        }
        
        // Calcular valor da comissão (apenas check-ins contam)
        $valor_comissao = floatval($total_checkins) * floatval($evento['valor_comissao']);
        
        error_log("Valor calculado: R$ " . number_format($valor_comissao, 2, ',', '.'));
        
        // Verificar se já existe comissão DISPONÍVEL para este promoter/evento
        $stmt = $pdo->prepare("SELECT id, valor, convidados_total, convidados_checkin FROM comissoes WHERE user_id = ? AND evento_id = ? AND status = 'disponivel'");
        $stmt->execute([$promoter_id, $evento_id]);
        $comissao_disponivel = $stmt->fetch();

        if ($comissao_disponivel) {
            // Atualizar comissão existente disponível
            $stmt = $pdo->prepare("
                UPDATE comissoes
                SET valor = ?, convidados_total = ?, convidados_checkin = ?
                WHERE user_id = ? AND evento_id = ? AND status = 'disponivel'
            ");
            $resultado = $stmt->execute([$valor_comissao, $total_convidados, $total_checkins, $promoter_id, $evento_id]);
            error_log("Comissão DISPONÍVEL ATUALIZADA: " . ($resultado ? "SIM" : "NÃO"));
        } else {
            // Verificar se existem comissões pagas/solicitadas para este evento
            $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM comissoes WHERE user_id = ? AND evento_id = ? AND status IN ('solicitado', 'pago')");
            $stmt->execute([$promoter_id, $evento_id]);
            $tem_comissoes_antigas = $stmt->fetch()['total'] > 0;

            if ($tem_comissoes_antigas) {
                // Se existem comissões já pagas/solicitadas, criar uma NOVA comissão para os novos check-ins
                // Calcular apenas o valor incremental (novos check-ins desde a última comissão)
                $stmt = $pdo->prepare("SELECT SUM(convidados_checkin) as total_ja_comissionado FROM comissoes WHERE user_id = ? AND evento_id = ?");
                $stmt->execute([$promoter_id, $evento_id]);
                $checkins_ja_comissionados = $stmt->fetch()['total_ja_comissionado'] ?: 0;

                $novos_checkins = $total_checkins - $checkins_ja_comissionados;
                $valor_nova_comissao = floatval($novos_checkins) * floatval($evento['valor_comissao']);

                if ($valor_nova_comissao > 0) {
                    $stmt = $pdo->prepare("
                        INSERT INTO comissoes (user_id, evento_id, valor, convidados_total, convidados_checkin, status)
                        VALUES (?, ?, ?, ?, ?, 'disponivel')
                    ");
                    $resultado = $stmt->execute([$promoter_id, $evento_id, $valor_nova_comissao, $total_convidados, $novos_checkins]);
                    error_log("NOVA COMISSÃO CRIADA (incremento): R$ " . number_format($valor_nova_comissao, 2) . " para $novos_checkins novos check-ins");
                }
            } else {
                // Primeira comissão para este evento
                $stmt = $pdo->prepare("
                    INSERT INTO comissoes (user_id, evento_id, valor, convidados_total, convidados_checkin, status)
                    VALUES (?, ?, ?, ?, ?, 'disponivel')
                ");
                $resultado = $stmt->execute([$promoter_id, $evento_id, $valor_comissao, $total_convidados, $total_checkins]);
                error_log("Comissão INSERIDA: " . ($resultado ? "SIM" : "NÃO"));
            }
        }
        
        // Atualizar pontuação do promoter
        $pontos = $total_checkins * 10; // 10 pontos por check-in
        $stmt = $pdo->prepare("UPDATE users SET pontuacao = pontuacao + 10 WHERE id = ?");
        $stmt->execute([$promoter_id]);
        error_log("Pontuação atualizada para promoter $promoter_id");
        
        return true;
    } catch (Exception $e) {
        error_log("ERRO em atualizarComissoes: " . $e->getMessage());
        return false;
    }
}

/**
 * Processar check-in e atualizar comissões automaticamente
 */
function processarCheckin($pdo, $qr_code) {
    // Buscar convidado
    $stmt = $pdo->prepare("
        SELECT c.*, e.id as evento_id, e.nome as evento_nome, u.nome as promoter_nome
        FROM convidados c
        INNER JOIN eventos e ON c.evento_id = e.id
        INNER JOIN users u ON c.promoter_id = u.id
        WHERE c.qr_code = ?
    ");
    $stmt->execute([$qr_code]);
    $convidado = $stmt->fetch();
    
    if (!$convidado) {
        return ['success' => false, 'message' => 'QR Code inválido'];
    }
    
    if ($convidado['checkin']) {
        return [
            'success' => false, 
            'message' => 'Check-in já realizado anteriormente',
            'data' => $convidado
        ];
    }
    
    // Realizar check-in
    $stmt = $pdo->prepare("UPDATE convidados SET checkin = 1, checkin_data = NOW() WHERE id = ?");
    $stmt->execute([$convidado['id']]);
    
    // Atualizar comissões automaticamente
    atualizarComissoes($pdo, $convidado['promoter_id'], $convidado['evento_id']);
    
    return [
        'success' => true,
        'message' => 'Check-in realizado com sucesso!',
        'data' => $convidado
    ];
}

/**
 * Enviar email com QR Code do convite
 * ✅ CORRIGIDO - Usa formatarData() do config.php
 */
function enviarEmailConvite($email_destino, $nome_convidado, $evento_info, $qr_code, $link_qrcode) {
    try {
        // Validar dados obrigatórios
        if (empty($email_destino) || empty($nome_convidado) || empty($qr_code)) {
            error_log("❌ ERRO: Dados obrigatórios faltando no envio de email");
            return false;
        }
        
        // Validar formato de email
        if (!filter_var($email_destino, FILTER_VALIDATE_EMAIL)) {
            error_log("❌ ERRO: Email inválido - " . $email_destino);
            return false;
        }
        
        // Configurações de email - usa constantes do config.php
        $remetente = EMAIL_REMETENTE;
        $nome_remetente = EMAIL_NOME_REMETENTE;
        
        // Assunto
        $assunto = "🎉 Seu ingresso para " . $evento_info['evento_nome'];
        
        // Data e hora formatadas
        $data_formatada = formatarData($evento_info['data']);
        $hora_formatada = date('H:i', strtotime($evento_info['hora']));
        
        // Corpo do email HTML
        $mensagem_html = '
        <!DOCTYPE html>
        <html lang="pt-BR">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <style>
                body {
                    font-family: Arial, sans-serif;
                    background-color: #f4f4f4;
                    margin: 0;
                    padding: 0;
                }
                .email-container {
                    max-width: 600px;
                    margin: 20px auto;
                    background: #ffffff;
                    border-radius: 10px;
                    overflow: hidden;
                    box-shadow: 0 4px 10px rgba(0,0,0,0.1);
                }
                .header {
                    background: linear-gradient(135deg, #FFD700, #FFA500);
                    color: #000;
                    padding: 30px;
                    text-align: center;
                }
                .header h1 {
                    margin: 0;
                    font-size: 28px;
                }
                .content {
                    padding: 30px;
                }
                .evento-nome {
                    font-size: 24px;
                    font-weight: bold;
                    color: #FFD700;
                    margin-bottom: 15px;
                    text-align: center;
                }
                .info-item {
                    margin: 10px 0;
                    font-size: 16px;
                    color: #333;
                }
                .qr-container {
                    text-align: center;
                    background: #f8f8f8;
                    padding: 30px;
                    border-radius: 10px;
                    margin: 20px 0;
                }
                .qr-code {
                    font-family: monospace;
                    font-size: 18px;
                    background: #000;
                    color: #FFD700;
                    padding: 15px;
                    border-radius: 8px;
                    display: inline-block;
                    margin-top: 15px;
                    word-break: break-all;
                }
                .btn {
                    display: inline-block;
                    padding: 15px 40px;
                    background: linear-gradient(135deg, #FFD700, #FFA500);
                    color: #000;
                    text-decoration: none;
                    border-radius: 8px;
                    font-weight: bold;
                    font-size: 16px;
                    margin: 20px 0;
                }
                .alert {
                    background: #fff3cd;
                    border: 2px solid #FFA500;
                    color: #856404;
                    padding: 15px;
                    border-radius: 8px;
                    margin: 20px 0;
                    line-height: 1.8;
                }
                .info-box {
                    background: #f8f8f8;
                    padding: 20px;
                    border-radius: 8px;
                    margin-top: 20px;
                    border-left: 4px solid #FFD700;
                }
                .info-box h3 {
                    color: #FFD700;
                    margin-bottom: 10px;
                }
                .info-box p {
                    color: #333;
                    line-height: 1.6;
                }
                .footer {
                    background: #1a1a1a;
                    color: #999;
                    padding: 20px;
                    text-align: center;
                    font-size: 14px;
                }
            </style>
        </head>
        <body>
            <div class="email-container">
                <div class="header">
                    <h1>🎉 Seu Ingresso está Pronto!</h1>
                </div>
                
                <div class="content">
                    <p style="font-size: 18px; color: #333;">
                        Olá, <strong>' . htmlspecialchars($nome_convidado) . '</strong>!
                    </p>
                    
                    <p style="font-size: 16px; color: #666; line-height: 1.6;">
                        Sua presença foi confirmada com sucesso! Abaixo estão os detalhes do seu ingresso:
                    </p>
                    
                    <div class="evento-nome">
                        ' . htmlspecialchars($evento_info['evento_nome']) . '
                    </div>
                    
                    <div class="info-item">
                        📅 <strong>Data:</strong> ' . $data_formatada . '
                    </div>
                    
                    <div class="info-item">
                        🕐 <strong>Horário:</strong> ' . $hora_formatada . '
                    </div>
                    
                    <div class="qr-container">
                        <h3 style="margin-bottom: 15px; color: #333;">Seu QR Code:</h3>
                        <div class="qr-code">' . htmlspecialchars($qr_code) . '</div>
                        <p style="margin-top: 20px; color: #666;">
                            <a href="' . htmlspecialchars($link_qrcode) . '" class="btn">
                                📥 BAIXAR INGRESSO
                            </a>
                        </p>
                    </div>
                    
                    <div class="alert">
                        <strong>⚠️ IMPORTANTE:</strong><br>
                        • Apresente este QR Code na entrada do evento<br>
                        • Salve este email ou baixe o ingresso<br>
                        • Tire um print/screenshot como backup
                    </div>
                    
                    ' . (!empty($evento_info['informacoes_importantes']) ? '
                    <div class="info-box">
                        <h3>📋 Informações Importantes:</h3>
                        <p>' . nl2br(htmlspecialchars($evento_info['informacoes_importantes'])) . '</p>
                    </div>
                    ' : '') . '
                    
                    <p style="margin-top: 30px; color: #666; font-size: 14px;">
                        Cadastro realizado por: <strong>' . htmlspecialchars($evento_info['promoter_nome']) . '</strong>
                    </p>
                </div>
                
                <div class="footer">
                    Desenvolvido por <strong>Alerson Araújo</strong><br>
                    Licenciado para <strong>Espaço Taal</strong>
                </div>
            </div>
        </body>
        </html>
        ';
        
        // Headers do email
        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
        $headers .= "From: " . $nome_remetente . " <" . $remetente . ">" . "\r\n";
        $headers .= "Reply-To: " . $remetente . "\r\n";
        $headers .= "X-Mailer: PHP/" . phpversion() . "\r\n";
        
        // Enviar email
        $enviado = mail($email_destino, $assunto, $mensagem_html, $headers);
        
        if ($enviado) {
            error_log("✅ Email enviado com sucesso para: " . $email_destino . " | QR Code: " . $qr_code);
            return true;
        } else {
            error_log("❌ ERRO ao enviar email para: " . $email_destino);
            return false;
        }
        
    } catch (Exception $e) {
        error_log("❌ EXCEÇÃO na função enviarEmailConvite: " . $e->getMessage());
        return false;
    }
}
?>