<?php
// includes/functions_notificacoes.php - Funções Auxiliares para Notificações v3.0

/**
 * Criar notificação genérica
 *
 * @param array $params Parâmetros da notificação
 * @return bool Sucesso da operação
 */
function criarNotificacao($params) {
    global $pdo;

    try {
        // LOG DE DEBUG
        error_log("[NOTIFICAÇÃO BASE] Iniciando criarNotificacao com params: " . json_encode($params));

        // Validar parâmetros obrigatórios
        $required = ['user_id', 'tipo', 'categoria', 'titulo', 'mensagem'];
        foreach ($required as $field) {
            if (!isset($params[$field])) {
                error_log("[NOTIFICAÇÃO BASE] ERRO: Campo obrigatório ausente: $field");
                throw new Exception("Campo obrigatório ausente: $field");
            }
        }

        // Preparar dados com valores padrão
        $data = [
            'user_id' => $params['user_id'],
            'tipo' => $params['tipo'],
            'categoria' => $params['categoria'],
            'titulo' => $params['titulo'],
            'mensagem' => $params['mensagem'],
            'referencia_tipo' => $params['referencia_tipo'] ?? null,
            'referencia_id' => $params['referencia_id'] ?? null,
            'link' => $params['link'] ?? null,
            'icon' => $params['icon'] ?? 'fa-bell',
            'cor' => $params['cor'] ?? '#FFD700'
        ];

        error_log("[NOTIFICAÇÃO BASE] Dados preparados: " . json_encode($data));

        // Inserir notificação
        $stmt = $pdo->prepare("
            INSERT INTO notificacoes (
                user_id, tipo, categoria, titulo, mensagem,
                referencia_tipo, referencia_id, link, icon, cor
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");

        error_log("[NOTIFICAÇÃO BASE] Executando INSERT...");

        $result = $stmt->execute([
            $data['user_id'],
            $data['tipo'],
            $data['categoria'],
            $data['titulo'],
            $data['mensagem'],
            $data['referencia_tipo'],
            $data['referencia_id'],
            $data['link'],
            $data['icon'],
            $data['cor']
        ]);

        if (!$result) {
            $errorInfo = $stmt->errorInfo();
            error_log("[NOTIFICAÇÃO BASE] ERRO SQL: " . json_encode($errorInfo));
            throw new Exception("Erro SQL: " . $errorInfo[2]);
        }

        error_log("[NOTIFICAÇÃO BASE] INSERT executado com sucesso. ID: " . $pdo->lastInsertId());

        // Atualizar contador do usuário
        error_log("[NOTIFICAÇÃO BASE] Atualizando contador do usuário {$data['user_id']}...");
        $pdo->prepare("
            UPDATE users
            SET notificacoes_nao_lidas = notificacoes_nao_lidas + 1
            WHERE id = ?
        ")->execute([$data['user_id']]);

        error_log("[NOTIFICAÇÃO BASE] Notificação criada com SUCESSO!");
        return true;

    } catch (Exception $e) {
        error_log("[NOTIFICAÇÃO BASE] EXCEÇÃO: " . $e->getMessage());
        error_log("[NOTIFICAÇÃO BASE] Stack trace: " . $e->getTraceAsString());
        return false;
    }
}

/**
 * Criar notificação de check-in
 *
 * @param int $promoter_id ID do promoter
 * @param array $convidado Dados do convidado
 * @param array $evento Dados do evento
 * @return bool
 */
function criarNotificacaoCheckin($promoter_id, $convidado, $evento = null) {
    global $pdo;

    try {
        // Se evento não foi passado, buscar
        if (!$evento && isset($convidado['evento_id'])) {
            $stmt = $pdo->prepare("SELECT * FROM eventos WHERE id = ?");
            $stmt->execute([$convidado['evento_id']]);
            $evento = $stmt->fetch(PDO::FETCH_ASSOC);
        }

        $nome_convidado = $convidado['nome'] ?? 'Convidado';
        $nome_evento = $evento['nome'] ?? 'evento';

        return criarNotificacao([
            'user_id' => $promoter_id,
            'tipo' => 'checkin',
            'categoria' => 'geral',
            'titulo' => 'Check-in Realizado',
            'mensagem' => "$nome_convidado fez check-in no evento $nome_evento!",
            'referencia_tipo' => 'convidado',
            'referencia_id' => $convidado['id'] ?? null,
            'link' => isset($evento['id']) ? "visualizar_evento.php?id={$evento['id']}" : null,
            'icon' => 'fa-user-check',
            'cor' => '#00C851'
        ]);

    } catch (Exception $e) {
        error_log("Erro ao criar notificação de check-in: " . $e->getMessage());
        return false;
    }
}

/**
 * Criar notificação de pagamento/comissão
 *
 * @param int $user_id ID do usuário
 * @param float $valor Valor do pagamento
 * @param string $tipo Tipo: comissao, pagamento, bonus
 * @param array $extra Dados extras (evento, descrição, etc)
 * @return bool
 */
function criarNotificacaoPagamento($user_id, $valor, $tipo = 'comissao', $extra = []) {
    $valor_formatado = 'R$ ' . number_format($valor, 2, ',', '.');

    $titulos = [
        'comissao' => 'Nova Comissão Recebida',
        'pagamento' => 'Pagamento Processado',
        'bonus' => 'Bônus Creditado'
    ];

    $mensagens = [
        'comissao' => "Você recebeu uma comissão de $valor_formatado!",
        'pagamento' => "Pagamento de $valor_formatado foi processado.",
        'bonus' => "Bônus de $valor_formatado creditado na sua conta!"
    ];

    $titulo = $titulos[$tipo] ?? 'Transação Financeira';
    $mensagem = $mensagens[$tipo] ?? "Transação de $valor_formatado realizada.";

    // Se houver descrição extra, adicionar
    if (isset($extra['descricao'])) {
        $mensagem .= " " . $extra['descricao'];
    }

    return criarNotificacao([
        'user_id' => $user_id,
        'tipo' => $tipo,
        'categoria' => 'financeiro',
        'titulo' => $titulo,
        'mensagem' => $mensagem,
        'referencia_tipo' => $extra['referencia_tipo'] ?? 'pagamento',
        'referencia_id' => $extra['referencia_id'] ?? null,
        'link' => $extra['link'] ?? 'financeiro.php',
        'icon' => 'fa-dollar-sign',
        'cor' => '#00C851'
    ]);
}

/**
 * Criar notificação de novo evento
 *
 * @param array $user_ids IDs dos usuários que devem receber
 * @param array $evento Dados do evento
 * @return int Número de notificações criadas
 */
function criarNotificacaoEvento($user_ids, $evento) {
    if (!is_array($user_ids)) {
        $user_ids = [$user_ids];
    }

    $nome_evento = $evento['nome'] ?? 'Novo Evento';
    $data_evento = isset($evento['data']) ? date('d/m/Y', strtotime($evento['data'])) : '';

    $mensagem = "O evento $nome_evento está disponível para você!";
    if ($data_evento) {
        $mensagem .= " Data: $data_evento";
    }

    $count = 0;
    foreach ($user_ids as $user_id) {
        if (criarNotificacao([
            'user_id' => $user_id,
            'tipo' => 'novo_evento',
            'categoria' => 'eventos',
            'titulo' => 'Novo Evento Disponível',
            'mensagem' => $mensagem,
            'referencia_tipo' => 'evento',
            'referencia_id' => $evento['id'] ?? null,
            'link' => isset($evento['id']) ? "eventos.php?destaque={$evento['id']}" : 'eventos.php',
            'icon' => 'fa-calendar-alt',
            'cor' => '#FFD700'
        ])) {
            $count++;
        }
    }

    return $count;
}

/**
 * Criar notificação de atualização de evento
 *
 * @param array $user_ids IDs dos usuários (promoters do evento)
 * @param array $evento Dados do evento
 * @param string $tipo_atualizacao cancelado, adiado, atualizado
 * @return int Número de notificações criadas
 */
function criarNotificacaoAtualizacaoEvento($user_ids, $evento, $tipo_atualizacao = 'atualizado') {
    if (!is_array($user_ids)) {
        $user_ids = [$user_ids];
    }

    $nome_evento = $evento['nome'] ?? 'Evento';

    $titulos = [
        'cancelado' => 'Evento Cancelado',
        'adiado' => 'Evento Adiado',
        'atualizado' => 'Evento Atualizado'
    ];

    $cores = [
        'cancelado' => '#dc3545',
        'adiado' => '#FFA500',
        'atualizado' => '#FFD700'
    ];

    $mensagens = [
        'cancelado' => "O evento $nome_evento foi cancelado.",
        'adiado' => "O evento $nome_evento foi adiado.",
        'atualizado' => "O evento $nome_evento teve suas informações atualizadas."
    ];

    $count = 0;
    foreach ($user_ids as $user_id) {
        if (criarNotificacao([
            'user_id' => $user_id,
            'tipo' => 'sistema',
            'categoria' => 'eventos',
            'titulo' => $titulos[$tipo_atualizacao] ?? 'Atualização de Evento',
            'mensagem' => $mensagens[$tipo_atualizacao] ?? "Evento $nome_evento atualizado.",
            'referencia_tipo' => 'evento',
            'referencia_id' => $evento['id'] ?? null,
            'link' => isset($evento['id']) ? "visualizar_evento.php?id={$evento['id']}" : 'eventos.php',
            'icon' => 'fa-calendar-alt',
            'cor' => $cores[$tipo_atualizacao] ?? '#FFD700'
        ])) {
            $count++;
        }
    }

    return $count;
}

/**
 * Criar notificação de saque
 *
 * @param int $user_id ID do usuário
 * @param string $status aprovado, recusado, processando
 * @param float $valor Valor do saque
 * @param string $motivo Motivo (para recusado)
 * @return bool
 */
function criarNotificacaoSaque($user_id, $status, $valor, $motivo = '') {
    $valor_formatado = 'R$ ' . number_format($valor, 2, ',', '.');

    $titulos = [
        'aprovado' => 'Saque Aprovado',
        'recusado' => 'Saque Recusado',
        'processando' => 'Saque em Processamento'
    ];

    $cores = [
        'aprovado' => '#00C851',
        'recusado' => '#dc3545',
        'processando' => '#FFA500'
    ];

    $mensagem = '';
    switch ($status) {
        case 'aprovado':
            $mensagem = "Seu saque de $valor_formatado foi aprovado e está sendo processado!";
            break;
        case 'recusado':
            $mensagem = "Seu saque de $valor_formatado foi recusado.";
            if ($motivo) {
                $mensagem .= " Motivo: $motivo";
            }
            break;
        case 'processando':
            $mensagem = "Seu saque de $valor_formatado está sendo processado.";
            break;
    }

    return criarNotificacao([
        'user_id' => $user_id,
        'tipo' => 'saque',
        'categoria' => 'financeiro',
        'titulo' => $titulos[$status] ?? 'Atualização de Saque',
        'mensagem' => $mensagem,
        'referencia_tipo' => 'saque',
        'referencia_id' => null,
        'link' => 'financeiro.php',
        'icon' => 'fa-money-bill-wave',
        'cor' => $cores[$status] ?? '#FFD700'
    ]);
}

/**
 * Criar notificação de nova mensagem
 *
 * @param int $destinatario_id ID do destinatário
 * @param int $mensagem_id ID da mensagem
 * @param string $assunto Assunto da mensagem
 * @param string $remetente_nome Nome do remetente
 * @return bool
 */
function criarNotificacaoMensagem($destinatario_id, $mensagem_id, $assunto, $remetente_nome) {
    return criarNotificacao([
        'user_id' => $destinatario_id,
        'tipo' => 'mensagem',
        'categoria' => 'mensagens',
        'titulo' => 'Nova Mensagem Recebida',
        'mensagem' => "$remetente_nome enviou: $assunto",
        'referencia_tipo' => 'mensagem',
        'referencia_id' => $mensagem_id,
        'link' => 'mensagens.php',
        'icon' => 'fa-envelope',
        'cor' => '#FFA500'
    ]);
}

/**
 * Criar notificação do sistema
 *
 * @param int $user_id ID do usuário
 * @param string $titulo Título da notificação
 * @param string $mensagem Mensagem
 * @param string $link Link opcional
 * @return bool
 */
function criarNotificacaoSistema($user_id, $titulo, $mensagem, $link = null) {
    return criarNotificacao([
        'user_id' => $user_id,
        'tipo' => 'sistema',
        'categoria' => 'geral',
        'titulo' => $titulo,
        'mensagem' => $mensagem,
        'referencia_tipo' => 'sistema',
        'referencia_id' => null,
        'link' => $link,
        'icon' => 'fa-info-circle',
        'cor' => '#17a2b8'
    ]);
}

/**
 * Criar notificações em lote
 *
 * @param array $user_ids Array de IDs de usuários
 * @param array $params Parâmetros da notificação (sem user_id)
 * @return int Número de notificações criadas com sucesso
 */
function criarNotificacoesEmLote($user_ids, $params) {
    if (!is_array($user_ids)) {
        return 0;
    }

    $count = 0;
    foreach ($user_ids as $user_id) {
        $params['user_id'] = $user_id;
        if (criarNotificacao($params)) {
            $count++;
        }
    }

    return $count;
}

/**
 * Marcar notificação como lida
 *
 * @param int $notificacao_id ID da notificação
 * @param int $user_id ID do usuário (para segurança)
 * @return bool
 */
function marcarNotificacaoLida($notificacao_id, $user_id) {
    global $pdo;

    try {
        $stmt = $pdo->prepare("
            UPDATE notificacoes
            SET lida = 1, lida_em = NOW()
            WHERE id = ? AND user_id = ? AND lida = 0
        ");
        $stmt->execute([$notificacao_id, $user_id]);

        if ($stmt->rowCount() > 0) {
            // Decrementar contador
            $pdo->prepare("
                UPDATE users
                SET notificacoes_nao_lidas = GREATEST(notificacoes_nao_lidas - 1, 0)
                WHERE id = ?
            ")->execute([$user_id]);

            return true;
        }

        return false;

    } catch (Exception $e) {
        error_log("Erro ao marcar notificação como lida: " . $e->getMessage());
        return false;
    }
}

/**
 * Marcar todas as notificações de um usuário como lidas
 *
 * @param int $user_id ID do usuário
 * @param string $categoria Categoria específica (opcional)
 * @return bool
 */
function marcarTodasNotificacoesLidas($user_id, $categoria = null) {
    global $pdo;

    try {
        $sql = "UPDATE notificacoes SET lida = 1, lida_em = NOW() WHERE user_id = ? AND lida = 0";
        $params = [$user_id];

        if ($categoria) {
            $sql .= " AND categoria = ?";
            $params[] = $categoria;
        }

        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);

        // Resetar contador
        $pdo->prepare("UPDATE users SET notificacoes_nao_lidas = 0 WHERE id = ?")
            ->execute([$user_id]);

        return true;

    } catch (Exception $e) {
        error_log("Erro ao marcar todas as notificações como lidas: " . $e->getMessage());
        return false;
    }
}

/**
 * Criar notificação para admins sobre nova solicitação de saque
 *
 * @param int $user_id ID do usuário que solicitou
 * @param float $valor Valor do saque
 * @param string $tipo pix, credito, cartao_consumo
 * @param array $dados_usuario Dados do usuário (opcional)
 * @return int Número de admins notificados
 */
function criarNotificacaoNovaSolicitacaoSaque($user_id, $valor, $tipo, $dados_usuario = null) {
    global $pdo;

    // LOG DE DEBUG
    error_log("[NOTIFICAÇÃO] Iniciando criarNotificacaoNovaSolicitacaoSaque - User: $user_id, Valor: $valor, Tipo: $tipo");

    try {
        // Se dados do usuário não foram passados, buscar
        if (!$dados_usuario) {
            $stmt = $pdo->prepare("SELECT nome, role FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $dados_usuario = $stmt->fetch();
        }

        $valor_formatado = 'R$ ' . number_format($valor, 2, ',', '.');
        $nome_usuario = $dados_usuario['nome'] ?? 'Usuário';

        // Ajustar labels de role
        $role_labels = [
            'user' => 'Promoter',
            'produtor' => 'Produtor',
            'promoter' => 'Promoter'
        ];
        $role_label = $role_labels[$dados_usuario['role']] ?? 'Usuário';

        // LOG DE DEBUG
        error_log("[NOTIFICAÇÃO] Nome: $nome_usuario, Role: {$dados_usuario['role']}, Label: $role_label");

        $titulos = [
            'pix' => 'Nova Solicitação de Saque PIX',
            'credito' => 'Nova Solicitação de Crédito de Consumo',
            'cartao_consumo' => 'Nova Solicitação de Cartão de Consumo'
        ];

        $mensagens = [
            'pix' => "$role_label $nome_usuario solicitou saque PIX de $valor_formatado",
            'credito' => "$role_label $nome_usuario solicitou crédito de consumo de $valor_formatado",
            'cartao_consumo' => "$role_label $nome_usuario solicitou cartão de consumo de $valor_formatado"
        ];

        $titulo = $titulos[$tipo] ?? 'Nova Solicitação de Saque';
        $mensagem = $mensagens[$tipo] ?? "$role_label $nome_usuario solicitou saque de $valor_formatado";

        // Buscar todos os admins
        $stmt = $pdo->query("SELECT id FROM users WHERE role = 'admin'");
        $admins = $stmt->fetchAll(PDO::FETCH_COLUMN);

        // LOG DE DEBUG
        error_log("[NOTIFICAÇÃO] Total de admins encontrados: " . count($admins));

        $count = 0;
        foreach ($admins as $admin_id) {
            $result = criarNotificacao([
                'user_id' => $admin_id,
                'tipo' => 'saque',
                'categoria' => 'financeiro',
                'titulo' => $titulo,
                'mensagem' => $mensagem,
                'referencia_tipo' => 'saque',
                'referencia_id' => $user_id,
                'link' => 'admin_saques.php',
                'icon' => 'fa-hand-holding-usd',
                'cor' => '#FFA500'
            ]);

            if ($result) {
                $count++;
                error_log("[NOTIFICAÇÃO] Notificação criada para admin ID: $admin_id");
            } else {
                error_log("[NOTIFICAÇÃO] FALHA ao criar notificação para admin ID: $admin_id");
            }
        }

        error_log("[NOTIFICAÇÃO] Total de notificações criadas: $count");
        return $count;

    } catch (Exception $e) {
        error_log("[NOTIFICAÇÃO] ERRO: " . $e->getMessage());
        error_log("[NOTIFICAÇÃO] Stack trace: " . $e->getTraceAsString());
        return 0;
    }
}

/**
 * Criar notificação para admins sobre nova mensagem recebida
 *
 * @param int $remetente_id ID do remetente
 * @param int $mensagem_id ID da mensagem
 * @param string $assunto Assunto da mensagem
 * @param string $remetente_nome Nome do remetente
 * @return int Número de admins notificados
 */
function criarNotificacaoMensagemParaAdmin($remetente_id, $mensagem_id, $assunto, $remetente_nome) {
    global $pdo;

    try {
        // Buscar todos os admins
        $stmt = $pdo->query("SELECT id FROM users WHERE role = 'admin'");
        $admins = $stmt->fetchAll(PDO::FETCH_COLUMN);

        $count = 0;
        foreach ($admins as $admin_id) {
            if (criarNotificacao([
                'user_id' => $admin_id,
                'tipo' => 'mensagem',
                'categoria' => 'mensagens',
                'titulo' => 'Nova Mensagem Recebida',
                'mensagem' => "$remetente_nome enviou: $assunto",
                'referencia_tipo' => 'mensagem',
                'referencia_id' => $mensagem_id,
                'link' => 'admin_mensagens.php',
                'icon' => 'fa-envelope',
                'cor' => '#FFA500'
            ])) {
                $count++;
            }
        }

        return $count;

    } catch (Exception $e) {
        error_log("Erro ao criar notificação de mensagem para admin: " . $e->getMessage());
        return 0;
    }
}

/**
 * Criar notificação para admins sobre nova inscrição em evento
 *
 * @param int $promoter_id ID do promoter
 * @param int $evento_id ID do evento
 * @param string $nome_promoter Nome do promoter
 * @param string $nome_evento Nome do evento
 * @return int Número de admins notificados
 */
function criarNotificacaoInscricaoEvento($promoter_id, $evento_id, $nome_promoter, $nome_evento) {
    global $pdo;

    try {
        // Buscar todos os admins
        $stmt = $pdo->query("SELECT id FROM users WHERE role = 'admin'");
        $admins = $stmt->fetchAll(PDO::FETCH_COLUMN);

        $count = 0;
        foreach ($admins as $admin_id) {
            if (criarNotificacao([
                'user_id' => $admin_id,
                'tipo' => 'sistema',
                'categoria' => 'eventos',
                'titulo' => 'Nova Inscrição em Evento',
                'mensagem' => "$nome_promoter se inscreveu no evento $nome_evento",
                'referencia_tipo' => 'evento',
                'referencia_id' => $evento_id,
                'link' => "admin_eventos.php?evento=$evento_id",
                'icon' => 'fa-user-plus',
                'cor' => '#00C851'
            ])) {
                $count++;
            }
        }

        return $count;

    } catch (Exception $e) {
        error_log("Erro ao criar notificação de inscrição em evento: " . $e->getMessage());
        return 0;
    }
}

/**
 * Criar notificação para admins sobre novo cadastro de promoter
 *
 * @param int $promoter_id ID do novo promoter
 * @param string $nome_promoter Nome do promoter
 * @return int Número de admins notificados
 */
function criarNotificacaoNovoCadastro($promoter_id, $nome_promoter) {
    global $pdo;

    try {
        // Buscar todos os admins
        $stmt = $pdo->query("SELECT id FROM users WHERE role = 'admin'");
        $admins = $stmt->fetchAll(PDO::FETCH_COLUMN);

        $count = 0;
        foreach ($admins as $admin_id) {
            if (criarNotificacao([
                'user_id' => $admin_id,
                'tipo' => 'sistema',
                'categoria' => 'geral',
                'titulo' => 'Novo Promoter Cadastrado',
                'mensagem' => "$nome_promoter se cadastrou e aguarda aprovação",
                'referencia_tipo' => 'user',
                'referencia_id' => $promoter_id,
                'link' => 'admin_usuarios.php',
                'icon' => 'fa-user-check',
                'cor' => '#17a2b8'
            ])) {
                $count++;
            }
        }

        return $count;

    } catch (Exception $e) {
        error_log("Erro ao criar notificação de novo cadastro: " . $e->getMessage());
        return 0;
    }
}

